// FILE: src/commands/app_cmds.rs
// ------------------------------

use crate::config::store::ConfigStore;
use crate::config::models::{AppConfig, HostConfig, InventoryNode, SerialDeviceInfo}; 
use crate::config::security;
use crate::connection::ssh_client::ClientHandler;
use crate::connection::sftp::SftpManager;
use crate::connection::ssh_session::HostConnection;
use crate::serial::models::SerialCommand;
use crate::terminal::pty::PtySession;
use tauri::State;
use tokio::sync::{mpsc, RwLock, Mutex};
use std::sync::{Arc, atomic::AtomicBool};
use russh::client::Handle;
use russh::ChannelId;
use uuid::Uuid;

pub struct AppState {
    pub config_store: ConfigStore,
    pub ssh_manager: crate::connection::manager::ConnectionManager,
    pub serial_sessions: Arc<RwLock<std::collections::HashMap<String, mpsc::Sender<SerialCommand>>>>,
    pub serial_read_buffers: Arc<RwLock<std::collections::HashMap<String, Vec<u8>>>>,
    pub serial_status: Arc<RwLock<std::collections::HashMap<String, String>>>,
    pub ssh_sessions: Arc<RwLock<std::collections::HashMap<String, (
        Arc<Handle<ClientHandler>>, 
        ChannelId, 
        mpsc::Sender<(u32, u32)>,
        Arc<HostConnection>
    )>>>,
    pub sftp_sessions: Arc<RwLock<std::collections::HashMap<String, Arc<Mutex<SftpManager>>>>>,
    pub sftp_cancel_tokens: Arc<RwLock<std::collections::HashMap<String, Arc<AtomicBool>>>>,
    pub local_ptys: Arc<RwLock<std::collections::HashMap<String, Arc<PtySession>>>>,
}

#[tauri::command]
pub async fn get_lock_status(state: State<'_, AppState>) -> Result<bool, String> {
    Ok(state.config_store.is_locked())
}

#[tauri::command]
pub async fn get_security_status() -> Result<bool, String> {
    Ok(security::is_key_file_encrypted())
}

#[tauri::command]
pub async fn unlock_app(state: State<'_, AppState>, password: String) -> Result<bool, String> {
    state.config_store.unlock(&password).map_err(|e| e.to_string())
}

#[tauri::command]
pub async fn set_master_password(state: State<'_, AppState>, _old_pass: Option<String>, new_pass: Option<String>) -> Result<(), String> {
    if state.config_store.is_locked() {
        return Err("Application is locked".into());
    }
    if let Some(ref p) = new_pass {
        if p.len() > 0 && p.len() < 4 { return Err("Password too short".into()); }
    }
    if let Some(key) = state.config_store.get_key() {
        security::set_master_password(&key, new_pass).map_err(|e| e.to_string())?;
        Ok(())
    } else {
        Err("Internal Error: Key not loaded".into())
    }
}

#[tauri::command]
pub async fn get_app_config(state: State<'_, AppState>) -> Result<AppConfig, String> {
    if state.config_store.is_locked() { return Err("LOCKED".to_string()); }
    Ok(state.config_store.get_snapshot())
}

#[tauri::command]
pub async fn save_app_config(state: State<'_, AppState>, config: AppConfig) -> Result<(), String> {
    state.config_store.update(|c| *c = config).map_err(|e| e.to_string())
}

#[tauri::command]
pub async fn save_ssh_host(
    state: State<'_, AppState>, name: String, host: String, user: String, pass: String, port: u16, color: Option<String>
) -> Result<(), String> {
    log::info!("Saving SSH Host: {} ({}@{}:{})", name, user, host, port);
    
    let new_id = Uuid::new_v4();
    let new_host = HostConfig {
        id: new_id, name, group: "Saved".into(), hostname: host, port, username: user,
        password: Some(pass), key_path: None, enable_monitoring: true, color: color,
    };
    
    state.config_store.update(|config| { 
        config.hosts.insert(new_id, new_host); 
    }).map_err(|e| {
        log::error!("Failed to update config store: {}", e);
        e.to_string()
    })
}

// BATCH 6 NEW: Save Serial Device Info
#[tauri::command]
pub async fn save_serial_device(
    state: State<'_, AppState>,
    instance_id: String,
    friendly_name: String,
    color: Option<String>
) -> Result<(), String> {
    state.config_store.update(|config| {
        let entry = config.serial_instances.entry(instance_id).or_insert(SerialDeviceInfo {
            friendly_name: "".into(),
            profile_id: None,
            color: None,
            auto_connect: false
        });
        entry.friendly_name = friendly_name;
        entry.color = color;
    }).map_err(|e| e.to_string())
}

#[tauri::command]
pub async fn delete_ssh_host(state: State<'_, AppState>, id: String) -> Result<(), String> {
    let uuid = Uuid::parse_str(&id).map_err(|_| "Invalid UUID")?;
    state.config_store.update(|config| { config.hosts.remove(&uuid); }).map_err(|e| e.to_string())
}

#[tauri::command]
pub async fn get_inventory_tree(state: State<'_, AppState>) -> Result<Vec<InventoryNode>, String> {
    if state.config_store.is_locked() { return Ok(vec![]); }
    let config = state.config_store.get_snapshot();
    let mut ssh_nodes = Vec::with_capacity(config.hosts.len());
    for (id, host) in config.hosts {
        ssh_nodes.push(InventoryNode { id: id.to_string(), label: host.name, kind: "ssh".into(), children: vec![] });
    }
    let mut serial_nodes = Vec::with_capacity(config.serial_profiles.len());
    for (id, profile) in config.serial_profiles {
        serial_nodes.push(InventoryNode { id: id.to_string(), label: profile.name, kind: "serial".into(), children: vec![] });
    }
    Ok(vec![
        InventoryNode { id: "root_ssh".into(), label: "SSH Hosts".into(), kind: "folder".into(), children: ssh_nodes },
        InventoryNode { id: "root_serial".into(), label: "Serial Profiles".into(), kind: "folder".into(), children: serial_nodes }
    ])
}

#[tauri::command]
pub async fn force_app_exit(app: tauri::AppHandle) { app.exit(0); }

#[tauri::command]
pub async fn cancel_sftp_transfer(state: State<'_, AppState>, host_id: String) -> Result<(), String> {
    let map = state.sftp_cancel_tokens.read().await;
    if let Some(token) = map.get(&host_id) {
        token.store(true, std::sync::atomic::Ordering::Relaxed);
        log::info!("Cancellation requested for host {}", host_id);
    }
    Ok(())
}