// FILE: src/commands/local_fs.rs
// ------------------------------
// Adapter to make Local Disk look like SFTP
// Updated to point to new sftp::io module for FileEntry
use crate::commands::sftp::io::FileEntry;
use std::time::UNIX_EPOCH;
use base64::{Engine as _, engine::general_purpose};

pub async fn list_files(path: &str) -> Result<Vec<FileEntry>, String> {
    let search_path = if path == "." || path.is_empty() { 
        std::env::current_dir().map_err(|e| e.to_string())? 
    } else { 
        std::path::PathBuf::from(path) 
    };

    let mut entries = tokio::fs::read_dir(search_path).await.map_err(|e| e.to_string())?;
    let mut file_list = Vec::new();

    while let Some(entry) = entries.next_entry().await.map_err(|e| e.to_string())? {
        let meta = entry.metadata().await.map_err(|e| e.to_string())?;
        
        let modified = meta.modified().unwrap_or(UNIX_EPOCH)
            .duration_since(UNIX_EPOCH).unwrap_or_default().as_secs();

        file_list.push(FileEntry {
            name: entry.file_name().to_string_lossy().to_string(),
            is_dir: meta.is_dir(),
            size: meta.len(),
            modified,
        });
    }

    file_list.sort_by(|a, b| b.is_dir.cmp(&a.is_dir).then(a.name.cmp(&b.name)));
    Ok(file_list)
}

pub async fn read_file(path: &str) -> Result<String, String> {
    let data = tokio::fs::read(path).await.map_err(|e| e.to_string())?;
    Ok(general_purpose::STANDARD.encode(&data))
}

pub async fn write_file(path: &str, content: &str) -> Result<(), String> {
    tokio::fs::write(path, content).await.map_err(|e| e.to_string())
}

pub async fn write_binary(path: &str, b64_data: &str) -> Result<(), String> {
    let bytes = general_purpose::STANDARD.decode(b64_data).map_err(|_| "Invalid Base64".to_string())?;
    tokio::fs::write(path, bytes).await.map_err(|e| e.to_string())
}

pub async fn create_entry(path: &str, is_dir: bool) -> Result<(), String> {
    if is_dir {
        tokio::fs::create_dir_all(path).await.map_err(|e| e.to_string())
    } else {
        tokio::fs::File::create(path).await.map_err(|e| e.to_string())?;
        Ok(())
    }
}

pub async fn delete(path: &str, is_dir: bool) -> Result<(), String> {
    if is_dir {
        tokio::fs::remove_dir_all(path).await.map_err(|e| e.to_string())
    } else {
        tokio::fs::remove_file(path).await.map_err(|e| e.to_string())
    }
}

pub async fn rename(src: &str, dst: &str) -> Result<(), String> {
    tokio::fs::rename(src, dst).await.map_err(|e| e.to_string())
}

// Local "Copy" is Read + Write
pub async fn copy(src: &str, dst: &str) -> Result<(), String> {
    // Check if dir
    let meta = tokio::fs::metadata(src).await.map_err(|e| e.to_string())?;
    if meta.is_dir() {
        return Err("Directory copy not supported locally yet".into());
    }
    tokio::fs::copy(src, dst).await.map_err(|e| e.to_string())?;
    Ok(())
}