// FILE: src/commands/serial_cmds.rs
// ---------------------------------

use crate::commands::app_cmds::AppState;
use crate::serial::{enumerator, supervisor::{SerialSupervisor, ConnectionState}, models::DeviceFingerprint, models::BytecodeConfig, models::SerialCommand};
use crate::config::models::SerialProfile;
use tauri::{State};
use tokio::sync::mpsc;
use uuid::Uuid;
use base64::{Engine as _, engine::general_purpose};
use serde::Serialize;
use std::time::Duration; // Added for timeout

const MAX_BUFFER_SIZE: usize = 1 * 1024 * 1024; // 1MB Safety Cap

#[derive(Serialize)]
pub struct SerialPollResponse {
    pub data: String, // Base64
    pub status: String, 
}

#[tauri::command]
pub async fn list_serial_devices() -> Result<Vec<DeviceFingerprint>, String> {
    tokio::task::spawn_blocking(enumerator::enumerate_devices)
        .await
        .map_err(|e| e.to_string())
}

#[tauri::command]
pub async fn start_serial_session(
    state: State<'_, AppState>,
    _profile_id: String,
    device_instance_id: String,
    session_id: String,
    baud_rate: Option<u32> 
) -> Result<String, String> { 
    
    let mut profile = SerialProfile {
        id: Uuid::new_v4(), name: "QC".into(), baud_rate: 115200, data_bits: 8,
        flow_control: "None".into(), stop_bits: 1, parity: "None".into(),
        tx_prefix_hex: None, tx_suffix_hex: None, force_crlf: false,
    };

    if let Some(b) = baud_rate {
        profile.baud_rate = b;
    }

    // BATCH 2 FIX: Port Probing to prevent double-connection panic loops
    // We try to open the port momentarily. If it fails (e.g. Access Denied), we assume it's busy.
    {
        // Resolve the port name from the Instance ID first
        let target = device_instance_id.clone();
        let port_name_opt = tokio::task::spawn_blocking(move || {
            let devices = enumerator::enumerate_devices();
            for dev in devices {
                if dev.instance_id == target { return Some(dev.port_name); }
            }
            None
        }).await.map_err(|e| e.to_string())?;

        if let Some(port_name) = port_name_opt {
             let baud = profile.baud_rate;
             // Attempt a quick open/close
             let probe_res = tokio::task::spawn_blocking(move || {
                 serialport::new(&port_name, baud)
                    .timeout(Duration::from_millis(100))
                    .open()
             }).await.map_err(|e| e.to_string())?;

             if let Err(e) = probe_res {
                 return Err(format!("Cannot open port (Busy or Access Denied): {}", e));
             }
             // If successful, the port closes here as the handle drops, freeing it for the Supervisor.
        }
    }

    // Capture state updates from Supervisor
    let (state_tx, mut state_rx) = mpsc::channel(10); 
    
    let (data_tx, mut data_rx) = mpsc::channel(100);
    let (write_tx, write_rx) = mpsc::channel::<SerialCommand>(100); 

    // Store Write Handle
    {
        let mut sessions = state.serial_sessions.write().await;
        sessions.insert(session_id.clone(), write_tx);
    }
    
    // Initialize Read Buffer
    {
        let mut buffers = state.serial_read_buffers.write().await;
        buffers.insert(session_id.clone(), Vec::new());
    }

    // Initialize Status
    {
        let mut statuses = state.serial_status.write().await;
        statuses.insert(session_id.clone(), "Searching".into());
    }

    // Start Supervisor
    let supervisor = SerialSupervisor::new(
        device_instance_id, profile.baud_rate, BytecodeConfig::default(),
        state_tx, data_tx, write_rx 
    );

    tokio::spawn(supervisor.run());

    // Task 1: Status Monitor
    let status_map = state.serial_status.clone();
    let sid_status = session_id.clone();
    
    tokio::spawn(async move {
        while let Some(new_state) = state_rx.recv().await {
            let str_state = match new_state {
                ConnectionState::Connected => "Connected",
                ConnectionState::Searching => "Searching",
                ConnectionState::Error(e) => {
                    log::warn!("Serial Error: {}", e); 
                    "Error" 
                },
                _ => "Disconnected"
            };
            
            let mut map = status_map.write().await;
            map.insert(sid_status.clone(), str_state.to_string());
        }
    });

    // Task 2: Data Buffer Monitor
    let buffer_map = state.serial_read_buffers.clone();
    let sid_data = session_id.clone();

    tokio::spawn(async move {
        while let Some(data) = data_rx.recv().await {
            let mut map = buffer_map.write().await;
            if let Some(buf) = map.get_mut(&sid_data) {
                if buf.len() + data.len() > MAX_BUFFER_SIZE {
                    buf.clear();
                    let warning = b"\n\n[...BUFFER OVERFLOW - CLEARED...]\n\n";
                    buf.extend_from_slice(warning);
                }
                buf.extend_from_slice(&data);
            }
        }
    });

    Ok("Started".into())
}

#[tauri::command]
pub async fn send_serial_data(state: State<'_, AppState>, session_id: String, data: String) -> Result<(), String> {
    let sessions = state.serial_sessions.read().await;
    if let Some(tx) = sessions.get(&session_id) {
        tx.send(SerialCommand::WriteData(data)).await.map_err(|_| "Closed".to_string())?;
        Ok(())
    } else {
        Err("Session not found".to_string())
    }
}

#[tauri::command]
pub async fn poll_serial(state: State<'_, AppState>, session_id: String) -> Result<SerialPollResponse, String> {
    // 1. Get Data
    let mut buf_map = state.serial_read_buffers.write().await;
    let b64 = if let Some(buf) = buf_map.get_mut(&session_id) {
        if buf.is_empty() {
            "".to_string()
        } else {
            let encoded = general_purpose::STANDARD.encode(&buf);
            buf.clear();
            encoded
        }
    } else {
        "".to_string()
    };

    // 2. Get Status (Batch 4 Fix)
    let status_map = state.serial_status.read().await;
    let status = status_map.get(&session_id).cloned().unwrap_or("Disconnected".to_string());

    Ok(SerialPollResponse {
        data: b64,
        status 
    })
}

#[tauri::command]
pub async fn toggle_serial_pin(
    state: State<'_, AppState>,
    session_id: String,
    pin: String,
    enabled: bool
) -> Result<(), String> {
    let sessions = state.serial_sessions.read().await;
    if let Some(tx) = sessions.get(&session_id) {
        let cmd = match pin.as_str() {
            "dtr" => SerialCommand::SetDtr(enabled),
            "rts" => SerialCommand::SetRts(enabled),
            _ => return Err("Invalid pin".into())
        };
        tx.send(cmd).await.map_err(|_| "Closed".to_string())?;
        Ok(())
    } else {
        Err("Session not found".to_string())
    }
}