use crate::commands::app_cmds::AppState;
use crate::commands::local_fs;
use crate::commands::sftp::session::get_sftp_session;
use crate::interop::drag::copy_to_clipboard;
use crate::interop::hook::enable_paste_hook;
use tauri::State;
use tauri_plugin_dialog::DialogExt;
use serde::Serialize;
use base64::{Engine as _, engine::general_purpose};
use tokio::io::AsyncWriteExt; 
use std::path::Path;
use uuid::Uuid;
use std::sync::{Arc, atomic::{AtomicBool, Ordering}};

#[derive(Serialize)]
pub struct FileEntry { 
    pub name: String, 
    pub is_dir: bool, 
    pub size: u64, 
    pub modified: u64 
}

#[derive(Serialize)]
pub struct FileContentResponse {
    pub data: String, 
    pub truncated: bool,
    pub size: u64,
}

#[tauri::command]
pub async fn sftp_list_files(state: State<'_, AppState>, host_id: String, path: String) -> Result<Vec<FileEntry>, String> {
    if host_id == "LOCAL" { return local_fs::list_files(&path).await; }
    let sftp = get_sftp_session(&state, &host_id).await?;
    let search_path = if path.trim().is_empty() { "." } else { &path };
    let entries = sftp.read_dir(search_path).await.map_err(|e| e.to_string())?;
    let mut file_list = Vec::new();
    for entry in entries {
        let attrs = entry.metadata();
        file_list.push(FileEntry {
            name: entry.file_name(), is_dir: attrs.is_dir(), 
            size: attrs.size.unwrap_or(0), modified: attrs.mtime.unwrap_or(0) as u64,
        });
    }
    file_list.sort_by(|a, b| b.is_dir.cmp(&a.is_dir).then(a.name.cmp(&b.name)));
    Ok(file_list)
}

#[tauri::command]
pub async fn sftp_read_file(state: State<'_, AppState>, host_id: String, path: String) -> Result<FileContentResponse, String> {
    if host_id == "LOCAL" { 
        let b64 = local_fs::read_file(&path).await?;
        return Ok(FileContentResponse { data: b64, truncated: false, size: 0 });
    }
    let sftp_guard = get_sftp_session(&state, &host_id).await?;
    let (bytes, truncated, size) = sftp_guard.fetch_for_edit(&path).await.map_err(|e| e.to_string())?;
    Ok(FileContentResponse {
        data: general_purpose::STANDARD.encode(&bytes),
        truncated,
        size
    })
}

#[tauri::command]
pub async fn sftp_write_file(state: State<'_, AppState>, host_id: String, path: String, content: String) -> Result<(), String> {
    if host_id == "LOCAL" { return local_fs::write_file(&path, &content).await; }
    let sftp = get_sftp_session(&state, &host_id).await?;
    let temp_dir = std::env::temp_dir();
    let temp_path = temp_dir.join(format!("up_{}", Uuid::new_v4()));
    {
        let mut file = tokio::fs::File::create(&temp_path).await.map_err(|e| e.to_string())?;
        file.write_all(content.as_bytes()).await.map_err(|e| e.to_string())?;
    }
    let res = sftp.upload_file(&temp_path, &path).await;
    let _ = tokio::fs::remove_file(temp_path).await;
    res.map_err(|e| e.to_string())
}

#[tauri::command]
pub async fn sftp_write_binary(state: State<'_, AppState>, host_id: String, path: String, base64_data: String) -> Result<(), String> {
    if host_id == "LOCAL" { return local_fs::write_binary(&path, &base64_data).await; }
    let cancel_token = {
        let mut map = state.sftp_cancel_tokens.write().await;
        map.entry(host_id.clone()).or_insert_with(|| Arc::new(AtomicBool::new(false))).clone()
    };
    if cancel_token.load(Ordering::Relaxed) { cancel_token.store(false, Ordering::Relaxed); }
    let sftp_guard = get_sftp_session(&state, &host_id).await?;
    let bytes = general_purpose::STANDARD.decode(&base64_data).map_err(|_| "Invalid Base64".to_string())?;
    let temp_path = std::env::temp_dir().join(format!("bin_{}", Uuid::new_v4()));
    {
        let mut file = tokio::fs::File::create(&temp_path).await.map_err(|e| e.to_string())?;
        file.write_all(&bytes).await.map_err(|e| e.to_string())?;
    }
    let res = sftp_guard.upload_file_cancellable(&temp_path, &path, cancel_token).await;
    let _ = tokio::fs::remove_file(temp_path).await;
    res.map_err(|e| e.to_string())
}

#[tauri::command]
pub async fn sftp_create_entry(state: State<'_, AppState>, host_id: String, path: String, is_dir: bool) -> Result<(), String> {
    if host_id == "LOCAL" { return local_fs::create_entry(&path, is_dir).await; }
    let sftp = get_sftp_session(&state, &host_id).await?;
    if is_dir { sftp.create_dir_robust(&path).await.map_err(|e| e.to_string()) } 
    else { sftp.create_empty(&path).await.map_err(|e| e.to_string()) }
}

#[tauri::command]
pub async fn sftp_delete(state: State<'_, AppState>, host_id: String, path: String, recursive: bool) -> Result<(), String> {
    if host_id == "LOCAL" { return local_fs::delete(&path, recursive).await; }
    let sftp = get_sftp_session(&state, &host_id).await?;
    sftp.delete(&path, recursive).await.map_err(|e| e.to_string())
}

#[tauri::command]
pub async fn sftp_rename(state: State<'_, AppState>, host_id: String, src: String, dst: String) -> Result<(), String> {
    if host_id == "LOCAL" { return local_fs::rename(&src, &dst).await; }
    let sftp = get_sftp_session(&state, &host_id).await?;
    sftp.rename(&src, &dst).await.map_err(|e| e.to_string())
}

#[tauri::command]
pub async fn sftp_copy(state: State<'_, AppState>, host_id: String, src: String, dst: String) -> Result<(), String> {
    if host_id == "LOCAL" { return local_fs::copy(&src, &dst).await; }
    let sftp = get_sftp_session(&state, &host_id).await?;
    sftp.copy(&src, &dst).await.map_err(|e| e.to_string())
}

#[tauri::command]
pub async fn sftp_to_clipboard(app: tauri::AppHandle, state: State<'_, AppState>, host_id: String, path: String) -> Result<(), String> {
    if host_id == "LOCAL" {
        let _ = app.run_on_main_thread(move || {
            if let Err(e) = copy_to_clipboard(std::path::Path::new(&path)) {
                log::error!("Clipboard Copy Failed: {:?}", e);
            } else { enable_paste_hook(); }
        });
        return Ok(());
    }
    let filename = Path::new(&path).file_name().map(|s| s.to_string_lossy().to_string()).unwrap_or_else(|| "download".to_string());
    let local_path = std::env::temp_dir().join(&filename);
    let local_path_clone = local_path.clone();
    {
        let sftp = get_sftp_session(&state, &host_id).await?;
        sftp.download_file(&path, &local_path_clone).await.map_err(|e| format!("Download failed: {}", e))?;
    }
    let _ = app.run_on_main_thread(move || {
        if let Err(e) = copy_to_clipboard(&local_path) { log::error!("Clipboard Copy Failed: {:?}", e); } 
        else { enable_paste_hook(); }
    });
    Ok(())
}

#[tauri::command]
pub async fn sftp_download(app: tauri::AppHandle, state: State<'_, AppState>, host_id: String, path: String) -> Result<(), String> {
    let filename = Path::new(&path).file_name().map(|s| s.to_string_lossy().to_string()).unwrap_or("dl".into());
    if host_id == "LOCAL" {
        if let Some(local) = app.dialog().file().set_file_name(&filename).blocking_save_file() {
            let dest_path = local.into_path().map_err(|e| e.to_string())?;
            local_fs::copy(&path, dest_path.to_str().unwrap()).await?;
        }
        return Ok(());
    }
    if let Some(local) = app.dialog().file().set_file_name(&filename).blocking_save_file() {
        let local_path = local.into_path().map_err(|e| e.to_string())?;
        let sftp = get_sftp_session(&state, &host_id).await?;
        sftp.download_file(&path, &local_path).await.map_err(|e| e.to_string())?;
    }
    Ok(())
}