use crate::commands::app_cmds::AppState;
use crate::connection::sftp::SftpManager;
use crate::connection::ssh_session::HostConnection;
use tauri::State;
use uuid::Uuid;
use std::sync::Arc;
use tokio::sync::{Mutex, OwnedMutexGuard};
use std::ops::{Deref, DerefMut};

// --- Helper Types ---

pub enum SftpSessionGuard {
    Persistent(OwnedMutexGuard<SftpManager>),
    Ephemeral(SftpManager),
}

impl Deref for SftpSessionGuard {
    type Target = SftpManager;
    fn deref(&self) -> &Self::Target {
        match self {
            SftpSessionGuard::Persistent(g) => g,
            SftpSessionGuard::Ephemeral(m) => m,
        }
    }
}

impl DerefMut for SftpSessionGuard {
    fn deref_mut(&mut self) -> &mut Self::Target {
        match self {
            SftpSessionGuard::Persistent(g) => g,
            SftpSessionGuard::Ephemeral(m) => m,
        }
    }
}

// --- Helper Functions ---

pub async fn get_sftp_session(state: &State<'_, AppState>, target_id: &str) -> Result<SftpSessionGuard, String> {
    // 1. Check Persistent SFTP Sessions
    {
        let map = state.sftp_sessions.read().await;
        if let Some(arc) = map.get(target_id) {
            let guard = arc.clone().lock_owned().await;
            return Ok(SftpSessionGuard::Persistent(guard));
        }
    }

    // 2. Check Active SSH Sessions (Piggybacking for Quick Connect)
    if target_id.starts_with("sess_") {
        let map = state.ssh_sessions.read().await;
        if let Some((handle, _, _, conn)) = map.get(target_id) {
            let manager = SftpManager::new(handle, conn.clone()).await
                .map_err(|e| format!("SFTP Piggyback failed: {}", e))?;
            return Ok(SftpSessionGuard::Ephemeral(manager));
        }
    }

    // 3. Fallback: Saved Profile (UUID) -> FORCE NEW CONNECTION
    let (_target_uuid, host_config) = if let Ok(u) = Uuid::parse_str(target_id) {
        let snapshot = state.config_store.get_snapshot();
        let cfg = snapshot.hosts.get(&u).cloned();
        (u, cfg)
    } else {
        (Uuid::nil(), None)
    };

    let cfg = host_config.ok_or_else(|| format!("Invalid Host ID or Session not found: {}", target_id))?;
    
    let conn = HostConnection::connect(&cfg).await
        .map_err(|e| format!("Dedicated SFTP Connection Failed: {}", e))?;
    let conn_arc = Arc::new(conn);

    let manager = SftpManager::new(&conn_arc.handle, conn_arc.clone()).await
        .map_err(|e| format!("SFTP Handshake Failed: {}", e))?;

    Ok(SftpSessionGuard::Ephemeral(manager))
}

// --- Commands ---

#[tauri::command]
pub async fn sftp_open_session(state: State<'_, AppState>, host_id: String) -> Result<String, String> {
    let (_target_uuid, host_config) = if let Ok(u) = Uuid::parse_str(&host_id) {
        let snapshot = state.config_store.get_snapshot();
        let cfg = snapshot.hosts.get(&u).cloned();
        (u, cfg)
    } else {
        (Uuid::nil(), None)
    };

    let cfg = host_config.ok_or("Host configuration not found")?;

    let conn = HostConnection::connect(&cfg).await
        .map_err(|e| format!("Connection failed: {}", e))?;
    let conn_arc = Arc::new(conn);

    let manager = SftpManager::new(&conn_arc.handle, conn_arc.clone()).await
        .map_err(|e| format!("SFTP Init failed: {}", e))?;

    let session_id = format!("sftp_{}", Uuid::new_v4());
    let mut map = state.sftp_sessions.write().await;
    map.insert(session_id.clone(), Arc::new(Mutex::new(manager)));

    Ok(session_id)
}

#[tauri::command]
pub async fn sftp_close_session(state: State<'_, AppState>, session_id: String) -> Result<(), String> {
    let mut map = state.sftp_sessions.write().await;
    map.remove(&session_id);
    Ok(())
}