// FILE: src/commands/sftp/sync.rs
// -------------------------------
// Native implementation of Sync logic (Two-Way Merge or One-Way Mirror)

use crate::commands::app_cmds::AppState;
use crate::commands::sftp::session::get_sftp_session;
use tauri::State;
use serde::{Serialize, Deserialize};
use std::path::Path;
use std::collections::{HashMap, HashSet};
use walkdir::WalkDir;
use russh_sftp::client::SftpSession;
use std::future::Future;
use std::pin::Pin;

#[derive(Debug, Serialize)]
pub struct SyncPlan {
    pub actions: Vec<SyncAction>,
    pub total_up_bytes: u64,
    pub total_down_bytes: u64,
}

#[derive(Debug, Serialize, Deserialize)]
pub struct SyncAction {
    pub rel_path: String,
    pub direction: String, // "Upload", "Download", "DeleteRemote", "CreateDir"
    pub reason: String,    // "New", "Newer", "Extraneous"
    pub size: u64,
}

#[derive(Clone)]
struct FileMeta {
    size: u64,
    mtime: u64,
    is_dir: bool,
}

#[tauri::command]
pub async fn sftp_sync_dry_run(
    state: State<'_, AppState>,
    host_id: String,
    local_path: String,
    remote_path: String,
    mode: String,
    excludes: Vec<String>
) -> Result<SyncPlan, String> {
    
    let sftp_guard = get_sftp_session(&state, &host_id).await?;
    let sftp = sftp_guard.session();

    let local_meta = scan_local(&local_path, &excludes).map_err(|e| format!("Local scan failed: {}", e))?;
    let remote_meta = scan_remote(sftp, &remote_path).await.map_err(|e| format!("Remote scan failed: {}", e))?;

    let mut actions = Vec::new();
    let mut total_up = 0;
    let mut total_down = 0;

    let all_paths: HashSet<&String> = local_meta.keys().chain(remote_meta.keys()).collect();

    for path in all_paths {
        let l = local_meta.get(path);
        let r = remote_meta.get(path);

        match (l, r) {
            (Some(lm), Some(rm)) => {
                if lm.is_dir { continue; }
                if lm.mtime > rm.mtime + 2 {
                    actions.push(SyncAction { rel_path: path.clone(), direction: "Upload".into(), reason: "Newer Local".into(), size: lm.size });
                    total_up += lm.size;
                } else if mode == "bidirectional" && rm.mtime > lm.mtime + 2 {
                    actions.push(SyncAction { rel_path: path.clone(), direction: "Download".into(), reason: "Newer Remote".into(), size: rm.size });
                    total_down += rm.size;
                } else if lm.size != rm.size {
                    actions.push(SyncAction { rel_path: path.clone(), direction: "Upload".into(), reason: "Size Mismatch".into(), size: lm.size });
                    total_up += lm.size;
                }
            },
            (Some(lm), None) => {
                if lm.is_dir { 
                    actions.push(SyncAction { rel_path: path.clone(), direction: "CreateDir".into(), reason: "New Dir".into(), size: 0 });
                } else {
                    actions.push(SyncAction { rel_path: path.clone(), direction: "Upload".into(), reason: "New".into(), size: lm.size });
                    total_up += lm.size;
                }
            },
            (None, Some(rm)) => {
                if rm.is_dir { continue; }
                if mode == "bidirectional" {
                    actions.push(SyncAction { rel_path: path.clone(), direction: "Download".into(), reason: "New Remote".into(), size: rm.size });
                    total_down += rm.size;
                } else {
                    actions.push(SyncAction { rel_path: path.clone(), direction: "DeleteRemote".into(), reason: "Extraneous".into(), size: 0 });
                }
            },
            (None, None) => {} 
        }
    }

    actions.sort_by(|a, b| a.rel_path.cmp(&b.rel_path));

    Ok(SyncPlan { actions, total_up_bytes: total_up, total_down_bytes: total_down })
}

#[tauri::command]
pub async fn sftp_sync_execute(
    state: State<'_, AppState>,
    host_id: String,
    local_base: String,
    remote_base: String,
    actions: Vec<SyncAction>
) -> Result<(), String> {
    
    let sftp_guard = get_sftp_session(&state, &host_id).await?;
    let sftp_manager = &*sftp_guard; 

    let mut created_remote_dirs = HashSet::new();

    for action in actions {
        let local_path = Path::new(&local_base).join(&action.rel_path);
        
        let remote_clean = remote_base.trim_end_matches('/');
        let rel_clean = action.rel_path.replace('\\', "/");
        let remote_path = format!("{}/{}", remote_clean, rel_clean);

        match action.direction.as_str() {
            "CreateDir" => {
                // Use new Robust method
                sftp_manager.create_dir_robust(&remote_path).await
                    .map_err(|e| format!("Failed to create dir {}: {}", action.rel_path, e))?;
                created_remote_dirs.insert(remote_path);
            },
            "Upload" => {
                if let Some(parent_idx) = remote_path.rfind('/') {
                    let parent = &remote_path[..parent_idx];
                    if !created_remote_dirs.contains(parent) {
                        // Use new Robust method
                        sftp_manager.create_dir_robust(parent).await
                            .map_err(|e| format!("Failed to create remote dir {}: {}", parent, e))?;
                        created_remote_dirs.insert(parent.to_string());
                    }
                }
                sftp_manager.upload_file(&local_path, &remote_path).await
                    .map_err(|e| format!("Failed to upload {}: {}", action.rel_path, e))?;
            },
            "Download" => {
                if let Some(parent) = local_path.parent() {
                    tokio::fs::create_dir_all(parent).await
                        .map_err(|e| format!("Failed to create local dir: {}", e))?;
                }
                sftp_manager.download_file(&remote_path, &local_path).await
                    .map_err(|e| format!("Failed to download {}: {}", action.rel_path, e))?;
            },
            "DeleteRemote" => {
                sftp_manager.delete(&remote_path, false).await
                    .map_err(|e| format!("Failed to delete remote {}: {}", action.rel_path, e))?;
            },
            _ => {}
        }
    }

    Ok(())
}

fn scan_local(base_path: &str, excludes: &[String]) -> Result<HashMap<String, FileMeta>, std::io::Error> {
    let mut map = HashMap::new();
    let root = Path::new(base_path);

    if !root.exists() {
        return Err(std::io::Error::new(std::io::ErrorKind::NotFound, "Local path not found"));
    }

    for entry in WalkDir::new(root).follow_links(true) {
        let entry = entry?;
        let path = entry.path();
        
        let path_str = path.to_string_lossy();
        if excludes.iter().any(|e| path_str.contains(e)) { continue; }

        let rel_path = path.strip_prefix(root).unwrap_or(path).to_string_lossy().replace('\\', "/");
        if rel_path.is_empty() { continue; }

        let meta = entry.metadata()?;
        map.insert(rel_path, FileMeta {
            size: meta.len(),
            mtime: meta.modified().unwrap_or(std::time::UNIX_EPOCH)
                .duration_since(std::time::UNIX_EPOCH).unwrap_or_default().as_secs(),
            is_dir: meta.is_dir(),
        });
    }
    Ok(map)
}

async fn scan_remote(sftp: &SftpSession, root_path: &str) -> Result<HashMap<String, FileMeta>, String> {
    let mut map = HashMap::new();
    let clean_root = root_path.trim_end_matches('/');
    
    fn walk<'a>(sftp: &'a SftpSession, current_abs: String, root_abs: String) 
    -> Pin<Box<dyn Future<Output = Result<Vec<(String, FileMeta)>, String>> + Send + 'a>> {
        Box::pin(async move {
            let mut results = Vec::new();
            let mut dir_stream = match sftp.read_dir(&current_abs).await {
                Ok(s) => s, Err(_) => return Ok(vec![])
            };

            while let Some(entry) = dir_stream.next() {
                let fname = entry.file_name();
                if fname == "." || fname == ".." { continue; }

                let full_path = format!("{}/{}", current_abs, fname);
                let rel_path = if full_path.starts_with(&root_abs) {
                    full_path[root_abs.len()..].trim_start_matches('/').to_string()
                } else { fname.clone() };

                let attrs = entry.metadata();
                let meta = FileMeta {
                    size: attrs.size.unwrap_or(0),
                    mtime: attrs.mtime.unwrap_or(0) as u64,
                    is_dir: attrs.is_dir(),
                };
                results.push((rel_path.clone(), meta.clone()));

                if meta.is_dir {
                    let sub_results = walk(sftp, full_path, root_abs.clone()).await?;
                    results.extend(sub_results);
                }
            }
            Ok(results)
        })
    }

    if sftp.metadata(clean_root).await.is_err() { return Ok(map); }
    let items = walk(sftp, clean_root.to_string(), clean_root.to_string()).await?;
    for (k, v) in items { map.insert(k, v); }
    Ok(map)
}