use crate::commands::app_cmds::AppState;
use crate::config::models::HostConfig;
use crate::monitor::engine::MetricsEngine;
use tauri::{State, Window, Emitter};
use uuid::Uuid;
use serde::{Deserialize, Serialize};
use tokio::sync::mpsc;
// Removed unused Arc import

#[derive(Deserialize)]
pub struct QuickConnectPayload {
    pub host: String,
    pub user: String,
    pub pass: String,
    pub port: u16,
}

#[derive(Serialize)]
pub struct SshPollResponse {
    pub data: String,
    pub metrics: Option<crate::monitor::models::HostStats>, 
}

#[derive(Serialize, Clone)]
struct SshDataPayload {
    id: String,
    data: Vec<u8>,
}

fn log_to_ui(window: &Window, msg: String) {
    let _ = window.emit("debug-log", msg);
}

#[tauri::command]
pub async fn resize_ssh_session(
    state: State<'_, AppState>,
    session_id: String,
    rows: u32,
    cols: u32
) -> Result<(), String> {
    let map = state.ssh_sessions.read().await;
    if let Some((_, _, resize_tx, _)) = map.get(&session_id) {
        let _ = resize_tx.send((rows, cols)).await;
        Ok(())
    } else {
        Err("Session not found".into())
    }
}

#[tauri::command]
pub async fn exec_ssh_silent(
    state: State<'_, AppState>,
    session_id: String,
    command: String
) -> Result<String, String> {
    let map = state.ssh_sessions.read().await;
    if let Some((handle, _, _, _)) = map.get(&session_id) {
        let mut channel = handle.channel_open_session().await
            .map_err(|e| format!("Failed to open channel: {}", e))?;
        
        channel.exec(true, command.as_bytes()).await
            .map_err(|e| format!("Exec failed: {}", e))?;
        
        let mut output = Vec::new();
        while let Some(msg) = channel.wait().await {
            match msg {
                russh::ChannelMsg::Data { data } => output.extend_from_slice(&data),
                russh::ChannelMsg::ExitStatus { .. } => break,
                _ => {}
            }
        }
        
        Ok(String::from_utf8_lossy(&output).to_string())
    } else {
        Err("Session not found".into())
    }
}

#[tauri::command]
pub async fn connect_ssh(
    window: Window,
    state: State<'_, AppState>,
    session_id: String,
    host_id: String,
    payload: Option<QuickConnectPayload> 
) -> Result<String, String> {
    
    // Fixed unused uuid warning
    let (_uuid, host_config) = if let Ok(u) = Uuid::parse_str(&host_id) {
        let config_snapshot = state.config_store.get_snapshot();
        let cfg = config_snapshot.hosts.get(&u).ok_or("Host config not found")?.clone();
        (u, cfg)
    } else {
        if let Some(p) = payload {
            let u = Uuid::nil(); 
            let cfg = HostConfig {
                id: u, name: format!("{}@{}", p.user, p.host), group: "Quick Connect".into(),
                hostname: p.host, port: p.port, username: p.user, password: Some(p.pass),
                key_path: None, enable_monitoring: true,
                color: None, 
            };
            (u, cfg)
        } else { return Err("Invalid Host ID".into()); }
    };

    log_to_ui(&window, format!("Connecting session {} to {}...", session_id, host_config.hostname));

    let connection = state.ssh_manager.get_terminal_connection(host_config.clone()).await
        .map_err(|e| format!("SSH Error: {}", e))?;

    let mut channel = connection.open_shell_channel().await
        .map_err(|e| format!("Channel Error: {}", e))?;
    
    let channel_id = channel.id();
    let client_handle = connection.handle.clone(); 
    
    let connection_store = connection.clone();

    let (resize_tx, mut resize_rx) = mpsc::channel::<(u32, u32)>(5);

    {
        let mut map = state.ssh_sessions.write().await;
        map.insert(session_id.clone(), (client_handle, channel_id, resize_tx, connection_store));
    }
    
    let engine_window = window.clone();
    let engine = MetricsEngine::new(session_id.clone(), connection.clone(), engine_window);
    tokio::spawn(engine.run_loop());

    let sid = session_id.clone();
    let data_window = window.clone(); 
    
    tokio::spawn(async move {
        loop {
            tokio::select! {
                Some((rows, cols)) = resize_rx.recv() => {
                    let _ = channel.window_change(cols, rows, 0, 0).await;
                }

                msg = channel.wait() => {
                    match msg {
                        Some(russh::ChannelMsg::Data { data }) => {
                            let payload = SshDataPayload {
                                id: sid.clone(),
                                data: data.to_vec()
                            };
                            if let Err(_) = data_window.emit("ssh-data", payload) {
                                break;
                            }
                        },
                        Some(russh::ChannelMsg::ExitStatus { .. }) | None => {
                            break;
                        }
                        _ => {} 
                    }
                }
            }
        }
    });

    Ok("Connected".into())
}

#[tauri::command]
pub async fn send_ssh_data(state: State<'_, AppState>, session_id: String, data: String) -> Result<(), String> {
    let mut map = state.ssh_sessions.write().await;
    if let Some((handle, channel_id, _, _)) = map.get_mut(&session_id) {
        handle.data(*channel_id, data.as_bytes().into()).await
            .map_err(|_| "Failed to write".to_string())?;
        Ok(())
    } else {
        Err("Session not found".into())
    }
}

#[tauri::command]
pub async fn poll_ssh(_state: State<'_, AppState>, _session_id: String) -> Result<SshPollResponse, String> {
    Ok(SshPollResponse {
        data: "".to_string(),
        metrics: None
    })
}

#[tauri::command]
pub async fn disconnect_ssh(state: State<'_, AppState>, session_id: String) -> Result<(), String> {
    {
        let mut map = state.ssh_sessions.write().await;
        map.remove(&session_id);
    }
    Ok(())
}