use crate::commands::app_cmds::AppState;
use tauri::State;
use serde::Serialize;
use std::process::Command;

#[derive(Serialize)]
pub struct ToolStatus {
    pub rsync: bool,
    pub sshpass: bool,
    pub local_os: String,
}

#[tauri::command]
pub async fn check_local_tools() -> Result<ToolStatus, String> {
    let rsync = check_command("rsync", &["--version"]);
    let sshpass = check_command("sshpass", &["-V"]);
    
    Ok(ToolStatus {
        rsync,
        sshpass,
        local_os: std::env::consts::OS.to_string(),
    })
}

fn check_command(cmd: &str, args: &[&str]) -> bool {
    Command::new(cmd)
        .args(args)
        .output()
        .map(|o| o.status.success())
        .unwrap_or(false)
}

#[derive(Serialize)]
pub struct RsyncDryRunResult {
    pub command: String,
    pub output: String,
}

// Kept for future native Rsync integration
#[allow(dead_code)]
pub async fn rsync_dry_run(
    state: State<'_, AppState>,
    src_host_id: Option<String>,
    src_path: String,
    dst_host_id: Option<String>,
    dst_path: String,
    excludes: Vec<String>
) -> Result<RsyncDryRunResult, String> {
    
    let snapshot = state.config_store.get_snapshot();
    
    let src_str = resolve_rsync_target(&snapshot, src_host_id, &src_path)?;
    let dst_str = resolve_rsync_target(&snapshot, dst_host_id, &dst_path)?;

    let mut args = vec![
        "-avz".to_string(),
        "--dry-run".to_string(),
        "--stats".to_string()
    ];
    
    for excl in excludes {
        args.push(format!("--exclude={}", excl));
    }
    
    args.push(src_str);
    args.push(dst_str);

    let full_cmd = format!("rsync {}", args.join(" "));

    let output = Command::new("rsync")
        .args(&args)
        .output()
        .map_err(|e| format!("Execution failed: {}", e))?;

    Ok(RsyncDryRunResult {
        command: full_cmd,
        output: String::from_utf8_lossy(&output.stdout).to_string(),
    })
}

#[allow(dead_code)]
fn resolve_rsync_target(
    config: &crate::config::models::AppConfig, 
    host_id: Option<String>, 
    path: &str
) -> Result<String, String> {
    if let Some(hid) = host_id {
        if hid == "local" {
            return Ok(path.to_string());
        }
        
        if let Ok(uuid) = uuid::Uuid::parse_str(&hid) {
            if let Some(host) = config.hosts.get(&uuid) {
                return Ok(format!("{}@{}:{}", host.username, host.hostname, path));
            }
        }
        return Err("Host config not found".into());
    }
    Ok(path.to_string())
}