// FILE: src/config/models.rs
// --------------------------

use serde::{Deserialize, Serialize};
use std::collections::HashMap;
use uuid::Uuid;

#[derive(Debug, Clone, Serialize, Deserialize)]
pub struct AppConfig {
    /// Map of UUID -> Host Configuration
    pub hosts: HashMap<Uuid, HostConfig>,
    
    /// Map of UUID -> Serial Configuration (Baud, Parity, etc.)
    pub serial_profiles: HashMap<Uuid, SerialProfile>,
    
    /// Global setting for blocked serial devices (hardware_id or instance_id)
    pub blocked_devices: Vec<String>,

    /// BATCH 4: Map InstanceID (Physical Hardware) -> User Settings
    /// Using #[serde(default)] allows loading old config files without this field
    #[serde(default)]
    pub serial_instances: HashMap<String, SerialDeviceInfo>,
}

impl Default for AppConfig {
    fn default() -> Self {
        Self {
            hosts: HashMap::new(),
            serial_profiles: HashMap::new(),
            blocked_devices: Vec::new(),
            serial_instances: HashMap::new(),
        }
    }
}

#[derive(Debug, Clone, Serialize, Deserialize)]
pub struct SerialDeviceInfo {
    pub friendly_name: String,
    pub profile_id: Option<Uuid>, // Link to a SerialProfile
    pub color: Option<String>,
    pub auto_connect: bool,
}

#[derive(Debug, Clone, Serialize, Deserialize)]
pub struct HostConfig {
    pub id: Uuid,
    pub name: String,
    pub group: String, 
    pub hostname: String,
    pub port: u16,
    pub username: String,
    pub password: Option<String>, 
    pub key_path: Option<String>,
    pub enable_monitoring: bool,
    pub color: Option<String>,
}

#[derive(Debug, Clone, Serialize, Deserialize)]
pub struct SerialProfile {
    pub id: Uuid,
    pub name: String,
    pub baud_rate: u32,
    pub data_bits: u8, 
    pub flow_control: String, 
    pub stop_bits: u8, 
    pub parity: String, 
    
    pub tx_prefix_hex: Option<String>,
    pub tx_suffix_hex: Option<String>,
    
    pub force_crlf: bool,
}

#[derive(Debug, Serialize)]
pub struct InventoryNode {
    pub id: String,
    pub label: String,
    pub kind: String, 
    pub children: Vec<InventoryNode>,
}