use aes_gcm::{
    aead::{Aead, KeyInit, Payload},
    Aes256Gcm, Key, Nonce
};
use rand::{RngCore, thread_rng};
use crate::config::models::AppConfig;
use crate::constants::get_key_path;
use std::io::{Read, Write};
use std::fs::File;
use std::path::Path;
use anyhow::{Result, Context, anyhow};
use serde::{Deserialize, Serialize};
use argon2::{
    password_hash::{
        rand_core::OsRng,
        SaltString 
    },
    Argon2
};

#[derive(Serialize, Deserialize)]
enum KeyEnvelope {
    Plain(Vec<u8>), 
    Encrypted {
        salt: String,
        nonce: Vec<u8>,
        ciphertext: Vec<u8>
    }
}

pub enum KeyLoadResult {
    Success([u8; 32]),
    NeedsPassword,
}

pub fn is_key_file_encrypted() -> bool {
    let key_path = get_key_path();
    if !key_path.exists() { return false; }

    if let Ok(file) = File::open(&key_path) {
        if file.metadata().map(|m| m.len()).unwrap_or(0) == 32 {
            return false;
        }
        let mut reader = std::io::BufReader::new(file);
        if let Ok(envelope) = serde_json::from_reader::<_, KeyEnvelope>(&mut reader) {
            return match envelope {
                KeyEnvelope::Plain(_) => false,
                KeyEnvelope::Encrypted { .. } => true,
            };
        }
    }
    false
}

pub fn load_master_key_status() -> Result<KeyLoadResult> {
    let key_path = get_key_path();
    
    if !key_path.exists() {
        let mut key = [0u8; 32];
        thread_rng().fill_bytes(&mut key);
        let envelope = KeyEnvelope::Plain(key.to_vec());
        let json = serde_json::to_vec(&envelope)?;
        let mut file = File::create(&key_path)?;
        file.write_all(&json)?;
        return Ok(KeyLoadResult::Success(key));
    }

    let mut file = File::open(&key_path)?;
    let mut buffer = Vec::new();
    file.read_to_end(&mut buffer)?;

    if buffer.len() == 32 {
        let mut arr = [0u8; 32];
        arr.copy_from_slice(&buffer);
        // Auto-migrate old legacy plain raw bytes to Envelope
        let envelope = KeyEnvelope::Plain(arr.to_vec());
        let json = serde_json::to_vec(&envelope)?;
        let mut w = File::create(&key_path)?;
        w.write_all(&json)?;
        return Ok(KeyLoadResult::Success(arr));
    }

    let envelope: KeyEnvelope = serde_json::from_slice(&buffer)
        .context("Invalid key file format")?;

    match envelope {
        KeyEnvelope::Plain(k) => {
            let mut arr = [0u8; 32];
            if k.len() != 32 { return Err(anyhow!("Corrupted key length")); }
            arr.copy_from_slice(&k);
            Ok(KeyLoadResult::Success(arr))
        },
        KeyEnvelope::Encrypted { .. } => Ok(KeyLoadResult::NeedsPassword),
    }
}

pub fn unlock_master_key(password: &str) -> Result<[u8; 32]> {
    let key_path = get_key_path();
    let mut file = File::open(&key_path)?;
    let mut buffer = Vec::new();
    file.read_to_end(&mut buffer)?;

    let envelope: KeyEnvelope = serde_json::from_slice(&buffer)?;

    match envelope {
        KeyEnvelope::Plain(k) => {
            let mut arr = [0u8; 32];
            arr.copy_from_slice(&k);
            Ok(arr)
        },
        KeyEnvelope::Encrypted { salt, nonce, ciphertext } => {
            let salt_obj = SaltString::from_b64(&salt).map_err(|e| anyhow!("{}", e))?;
            let argon2 = Argon2::default();
            
            let mut output_key_material = [0u8; 32];
            
            argon2.hash_password_into(
                password.as_bytes(), 
                salt_obj.as_salt().as_str().as_bytes(), 
                &mut output_key_material
            ).map_err(|e| anyhow!("{}", e))?;

            let key = Key::<Aes256Gcm>::from_slice(&output_key_material);
            let cipher = Aes256Gcm::new(key);
            let nonce_obj = Nonce::from_slice(&nonce);

            let plaintext = cipher.decrypt(nonce_obj, Payload {
                msg: &ciphertext,
                aad: b"OmniTerminalKey",
            }).map_err(|_| anyhow!("Invalid Password"))?;

            let mut final_key = [0u8; 32];
            final_key.copy_from_slice(&plaintext);
            Ok(final_key)
        }
    }
}

pub fn set_master_password(current_key: &[u8; 32], password: Option<String>) -> Result<()> {
    let envelope = if let Some(pass) = password {
        if pass.trim().is_empty() {
             log::info!("Master Password Set: None (Plain)");
             KeyEnvelope::Plain(current_key.to_vec())
        } else {
            log::info!("Master Password Set: Encrypted");
            let salt = SaltString::generate(&mut OsRng);
            let argon2 = Argon2::default();
            let mut kek = [0u8; 32];
            
            argon2.hash_password_into(
                pass.as_bytes(), 
                salt.as_salt().as_str().as_bytes(), 
                &mut kek
            ).map_err(|e| anyhow!("{}", e))?;

            let key = Key::<Aes256Gcm>::from_slice(&kek);
            let cipher = Aes256Gcm::new(key);
            
            let mut nonce_bytes = [0u8; 12];
            thread_rng().fill_bytes(&mut nonce_bytes);
            let nonce = Nonce::from_slice(&nonce_bytes);

            let ciphertext = cipher.encrypt(nonce, Payload {
                msg: current_key,
                aad: b"OmniTerminalKey",
            }).map_err(|e| anyhow!("{}", e))?;

            KeyEnvelope::Encrypted {
                salt: salt.to_string(),
                nonce: nonce_bytes.to_vec(),
                ciphertext
            }
        }
    } else {
        log::info!("Master Password Removed: Plain");
        KeyEnvelope::Plain(current_key.to_vec())
    };

    let json = serde_json::to_vec(&envelope)?;
    let key_path = get_key_path();
    let mut file = File::create(key_path)?;
    file.write_all(&json)?;
    
    Ok(())
}

// --- Inventory Logic ---

pub fn save_inventory(config: &AppConfig, path: &Path, master_key: &[u8; 32]) -> Result<()> {
    let json_data = serde_json::to_vec(config)?;
    let compressed_data = zstd::encode_all(&json_data[..], 3)?;

    let key = Key::<Aes256Gcm>::from_slice(master_key);
    let cipher = Aes256Gcm::new(key);
    
    let mut nonce_bytes = [0u8; 12];
    thread_rng().fill_bytes(&mut nonce_bytes);
    let nonce = Nonce::from_slice(&nonce_bytes);

    let ciphertext = cipher.encrypt(nonce, Payload {
        msg: &compressed_data,
        aad: b"OmniTerminalInventory", 
    }).map_err(|e| anyhow!("Encryption failure: {}", e))?;

    let mut file = File::create(path).context("Failed to create inventory file")?;
    file.write_all(&nonce_bytes)?;
    file.write_all(&ciphertext)?;

    Ok(())
}

pub fn load_inventory(path: &Path, master_key: &[u8; 32]) -> Result<AppConfig> {
    if !path.exists() {
        return Ok(AppConfig::default());
    }

    let mut file = File::open(path)?;
    let mut buffer = Vec::new();
    file.read_to_end(&mut buffer)?;

    if buffer.len() < 12 {
        return Ok(AppConfig::default()); 
    }

    let (nonce_bytes, ciphertext) = buffer.split_at(12);
    let nonce = Nonce::from_slice(nonce_bytes);

    let key = Key::<Aes256Gcm>::from_slice(master_key);
    let cipher = Aes256Gcm::new(key);

    let compressed_data = cipher.decrypt(nonce, Payload {
        msg: ciphertext,
        aad: b"OmniTerminalInventory",
    }).map_err(|_| anyhow!("Invalid key or corrupted inventory file"))?;

    let json_data = zstd::decode_all(&compressed_data[..])?;
    let config: AppConfig = serde_json::from_slice(&json_data)?;

    Ok(config)
}