use crate::config::models::AppConfig;
use crate::config::security::{self, KeyLoadResult};
use crate::constants::get_config_path;
use anyhow::{Result, anyhow};
use std::sync::{Arc, RwLock};

#[derive(Clone)]
pub struct ConfigStore {
    inner: Arc<RwLock<Option<AppConfig>>>,
    key: Arc<RwLock<Option<[u8; 32]>>>,
}

impl ConfigStore {
    pub fn load() -> Self {
        match security::load_master_key_status() {
            Ok(KeyLoadResult::Success(k)) => {
                let path = get_config_path();
                let config = match security::load_inventory(&path, &k) {
                    Ok(c) => c,
                    Err(e) => {
                        log::warn!("Failed to decrypt inventory: {}. Using default.", e);
                        AppConfig::default()
                    }
                };
                Self {
                    inner: Arc::new(RwLock::new(Some(config))),
                    key: Arc::new(RwLock::new(Some(k))),
                }
            },
            Ok(KeyLoadResult::NeedsPassword) => {
                log::info!("App Locked. Waiting for password.");
                Self {
                    inner: Arc::new(RwLock::new(None)),
                    key: Arc::new(RwLock::new(None)),
                }
            },
            Err(e) => {
                log::error!("Critical Key Error: {}", e);
                Self {
                    inner: Arc::new(RwLock::new(None)),
                    key: Arc::new(RwLock::new(None)),
                }
            }
        }
    }

    pub fn unlock(&self, password: &str) -> Result<bool> {
        let key_bytes = security::unlock_master_key(password)?;
        
        let path = get_config_path();
        let config = security::load_inventory(&path, &key_bytes)?;
        
        {
            let mut k_guard = self.key.write().unwrap();
            *k_guard = Some(key_bytes);
        }
        {
            let mut c_guard = self.inner.write().unwrap();
            *c_guard = Some(config);
        }
        
        Ok(true)
    }

    pub fn is_locked(&self) -> bool {
        self.inner.read().unwrap().is_none()
    }
    
    pub fn get_key(&self) -> Option<[u8; 32]> {
        *self.key.read().unwrap()
    }

    #[allow(dead_code)]
    pub fn save(&self) -> Result<()> {
        let path = get_config_path();
        let c_guard = self.inner.read().unwrap();
        let k_guard = self.key.read().unwrap();
        
        if let (Some(conf), Some(k)) = (c_guard.as_ref(), k_guard.as_ref()) {
             security::save_inventory(conf, &path, k)?;
             Ok(())
        } else {
            Err(anyhow!("Cannot save while locked"))
        }
    }

    pub fn get_snapshot(&self) -> AppConfig {
        let guard = self.inner.read().unwrap();
        if let Some(c) = guard.as_ref() {
            c.clone()
        } else {
            AppConfig::default() 
        }
    }

    pub fn update<F>(&self, modify_fn: F) -> Result<()> 
    where F: FnOnce(&mut AppConfig) 
    {
        let mut c_guard = self.inner.write().unwrap();
        
        if let Some(conf) = c_guard.as_mut() {
            modify_fn(conf);
            
            let k_guard = self.key.read().unwrap();
            if let Some(k) = k_guard.as_ref() {
                let path = get_config_path();
                security::save_inventory(conf, &path, k)?;
            }
            Ok(())
        } else {
            Err(anyhow!("Cannot update while locked"))
        }
    }
}