use crate::serial::{enumerator, stream::SerialStream, models::BytecodeConfig};
use serialport::SerialPortType;
use tokio::sync::mpsc;
use tokio::time::{sleep, Duration};
use std::sync::{Arc, Mutex};
use log::{info, warn, error};

#[derive(Debug, Clone)]
pub enum ConnectionState {
    Connected,
    Disconnected,
    Searching, // The "Graceful" state: polling specifically for the lost device
    Error(String),
}

/// The Supervisor manages the lifecycle of a serial connection.
/// It owns the stream, handles drops, and performs the auto-reconnect logic.
pub struct SerialSupervisor {
    target_instance_id: String, // The unique hardware ID we want to reconnect to
    baud_rate: u32,
    config: BytecodeConfig,
    state_tx: mpsc::Sender<ConnectionState>, // To update UI
    data_tx: mpsc::Sender<Vec<u8>>, // To send RX bytes to UI
    
    // The active stream is wrapped in a Mutex so we can replace it on reconnect
    current_stream: Arc<Mutex<Option<SerialStream>>>,
}

impl SerialSupervisor {
    pub fn new(
        target_instance_id: String,
        baud_rate: u32,
        config: BytecodeConfig,
        state_tx: mpsc::Sender<ConnectionState>,
        data_tx: mpsc::Sender<Vec<u8>>,
    ) -> Self {
        Self {
            target_instance_id,
            baud_rate,
            config,
            state_tx,
            data_tx,
            current_stream: Arc::new(Mutex::new(None)),
        }
    }

    /// Writes data to the currently active stream.
    /// Returns error if disconnected.
    pub async fn send_data(&self, data: &str) -> Result<(), String> {
        let stream_opt = self.current_stream.lock().unwrap();
        if let Some(stream) = stream_opt.as_ref() {
            // This is async inside the stream implementation
            stream.write_data(data).await
        } else {
            Err("Device Disconnected".to_string())
        }
    }

    /// Starts the supervisor loop. This should be spawned as a Tokio task.
    pub async fn run(self) {
        let _ = self.state_tx.send(ConnectionState::Searching).await;

        loop {
            // 1. Attempt Connection
            if let Some(port_name) = self.find_port_by_instance().await {
                match self.connect(&port_name) {
                    Ok(stream) => {
                        info!("Connected to {} ({})", port_name, self.target_instance_id);
                        let _ = self.state_tx.send(ConnectionState::Connected).await;
                        
                        // Start the reader loop
                        stream.spawn_reader(self.data_tx.clone());
                        
                        // Store the stream
                        {
                            let mut lock = self.current_stream.lock().unwrap();
                            *lock = Some(stream);
                        }

                        // 2. Monitor Loop (Passive)
                        // In a real scenario, we'd watch the handle. 
                        // Here, we periodically check if the port is still valid.
                        // Real implementation would rely on the Reader thread sending a "Break" signal.
                        // For this snippet, we wait until the Reader reports error or we poll validity.
                        self.monitor_connection().await;
                    }
                    Err(e) => {
                        warn!("Connection failed: {}", e);
                        let _ = self.state_tx.send(ConnectionState::Error(e.to_string())).await;
                        sleep(Duration::from_secs(2)).await;
                    }
                }
            } else {
                // Device not found
                // let _ = self.state_tx.send(ConnectionState::Searching).await; // Optional: reduces spam
                sleep(Duration::from_secs(1)).await;
            }
        }
    }

    /// Polls the Windows API via `enumerator` to find the COM port associated with our Instance ID.
    async fn find_port_by_instance(&self) -> Option<String> {
        // This is a blocking operation, run it on a thread
        let target = self.target_instance_id.clone();
        tokio::task::spawn_blocking(move || {
            let devices = enumerator::enumerate_devices();
            for dev in devices {
                // We match on Instance ID (robust) or Hardware ID (class)
                if dev.instance_id == target {
                    return Some(dev.port_name);
                }
            }
            None
        }).await.unwrap_or(None)
    }

    fn connect(&self, port_name: &str) -> Result<SerialStream, serialport::Error> {
        let port = serialport::new(port_name, self.baud_rate)
            .timeout(Duration::from_millis(100))
            .open()?;
        
        Ok(SerialStream::new(port, self.config.clone()))
    }

    /// Watches the active connection. If it dies, we clear the stream and return to main loop.
    async fn monitor_connection(&self) {
        // We simply wait. The Reader thread (in stream.rs) will close the channel if it errors.
        // Or we can poll `enumerator` every few seconds to see if the device vanished.
        loop {
            sleep(Duration::from_secs(1)).await;
            
            // Check if device is still present in Windows Device Manager
            // This is the "Supervisor" logic: The OS tells us it's gone before the stream fails usually.
            if self.find_port_by_instance().await.is_none() {
                warn!("Device {} vanished. Re-entering search mode.", self.target_instance_id);
                let _ = self.state_tx.send(ConnectionState::Searching).await;
                
                // Drop the stream to release the handle (if Windows hasn't forcibly done so)
                let mut lock = self.current_stream.lock().unwrap();
                *lock = None;
                return; // Return to main outer loop
            }
        }
    }
}