// FILE: src/connection/manager.rs
// -------------------------------

use crate::connection::ssh_session::HostConnection;
use crate::config::models::HostConfig;
use std::collections::HashMap;
use std::sync::Arc;
use tokio::sync::RwLock;
use uuid::Uuid;
use anyhow::Result;

#[derive(Clone)]
pub struct ConnectionManager {
    // PUBLIC: needed for SFTP "is connected" check
    pub connections: Arc<RwLock<HashMap<Uuid, Vec<Arc<HostConnection>>>>>,
}

impl ConnectionManager {
    pub fn new() -> Self {
        Self {
            connections: Arc::new(RwLock::new(HashMap::new())),
        }
    }

    pub async fn get_terminal_connection(&self, config: HostConfig) -> Result<Arc<HostConnection>> {
        // 1. Fast Path: Check if connection exists (Read Lock)
        {
            let map = self.connections.read().await;
            if let Some(pool) = map.get(&config.id) {
                if let Some(conn) = pool.first() {
                    // TODO: Check if conn is actually alive/connected?
                    return Ok(conn.clone());
                }
            }
        }

        // 2. Slow Path: Connect WITHOUT holding the lock
        // This prevents freezing the UI or other connections while this handshake happens.
        log::info!("Connecting Primary (Terminal) for {}", config.name);
        
        // Note: There is a small race condition here where two requests for the same host 
        // might both initiate a connection. We resolve this in step 3.
        let new_conn = HostConnection::connect(&config).await?;
        let shared_conn = Arc::new(new_conn);

        // 3. Write Path: Insert and deduplicate
        {
            let mut map = self.connections.write().await;
            
            // Check one last time if someone else beat us to it
            if let Some(pool) = map.get(&config.id) {
                if let Some(existing) = pool.first() {
                    log::info!("Race condition detected: Using connection established by another thread.");
                    return Ok(existing.clone());
                    // 'shared_conn' (our new one) drops here and disconnects gracefully
                }
            }

            map.insert(config.id, vec![shared_conn.clone()]);
        }
        
        Ok(shared_conn)
    }
}