
use russh_sftp::client::SftpSession;
use russh_sftp::client::fs::DirEntry;
use tokio::io::{AsyncReadExt, AsyncWriteExt};
use anyhow::{Result, anyhow};
use std::future::Future;
use std::pin::Pin;
use crate::constants::{MAX_EDIT_SIZE_BYTES, PREVIEW_SIZE_BYTES};

pub async fn read_dir(sftp: &SftpSession, path: &str) -> Result<Vec<DirEntry>> {
    let mut dir_stream = sftp.read_dir(path).await
        .map_err(|e| anyhow!("ReadDir init failed: {}", e))?;
    
    let mut entries = Vec::new();
    while let Some(entry) = dir_stream.next() {
        entries.push(entry);
    }
    Ok(entries)
}

pub async fn create_dir_all(sftp: &SftpSession, path: &str) -> Result<()> {
    let is_absolute = path.starts_with('/');
    let parts: Vec<&str> = path.split('/').filter(|s| !s.is_empty()).collect();
    
    let mut current = String::new();
    if is_absolute { current.push('/'); }

    for part in parts {
        if !current.ends_with('/') && !current.is_empty() { 
            current.push('/'); 
        }
        current.push_str(part);

        if sftp.metadata(&current).await.is_err() {
            if let Err(e) = sftp.create_dir(&current).await {
                if sftp.metadata(&current).await.is_err() {
                    return Err(anyhow!("Failed to create directory '{}': {}", current, e));
                }
            }
        }
    }
    Ok(())
}

pub async fn create_empty(sftp: &SftpSession, path: &str) -> Result<()> {
    let mut file = sftp.create(path).await
        .map_err(|e| anyhow!("Create file failed: {}", e))?;
    file.shutdown().await?;
    Ok(())
}

pub async fn delete(sftp: &SftpSession, path: &str, recursive: bool) -> Result<()> {
    let stat = sftp.metadata(path).await
        .map_err(|e| anyhow!("Stat failed: {}", e))?;
    
    if stat.is_dir() {
        if recursive {
            delete_recursive(sftp, path).await
        } else {
            sftp.remove_dir(path).await.map_err(|e| anyhow!("{}", e))
        }
    } else {
        sftp.remove_file(path).await.map_err(|e| anyhow!("Remove File failed: {}", e))
    }
}

fn delete_recursive<'a>(sftp: &'a SftpSession, path: &'a str) -> Pin<Box<dyn Future<Output = Result<()>> + Send + 'a>> {
    Box::pin(async move {
        let mut dir_stream = sftp.read_dir(path).await?;
        while let Some(entry) = dir_stream.next() {
            let filename = entry.file_name();
            if filename == "." || filename == ".." { continue; }
            let sub_path = if path.ends_with('/') { format!("{}{}", path, filename) } else { format!("{}/{}", path, filename) };
            let meta = entry.metadata();
            if meta.is_dir() { 
                delete_recursive(sftp, &sub_path).await?; 
            } else { 
                sftp.remove_file(&sub_path).await?; 
            }
        }
        sftp.remove_dir(path).await?;
        Ok(())
    })
}

pub async fn rename(sftp: &SftpSession, src: &str, dst: &str) -> Result<()> {
    sftp.rename(src, dst).await.map_err(|e| anyhow!("Rename failed: {}", e))
}

pub async fn copy(sftp: &SftpSession, src: &str, dst: &str) -> Result<()> {
    let stat = sftp.metadata(src).await.map_err(|e| anyhow!("Source not found: {}", e))?;
    if stat.is_dir() { 
        copy_recursive(sftp, src, dst).await 
    } else { 
        copy_file(sftp, src, dst).await 
    }
}

async fn copy_file(sftp: &SftpSession, src: &str, dst: &str) -> Result<()> {
    let mut src_file = sftp.open(src).await.map_err(|e| anyhow!("Failed to open source: {}", e))?;
    let mut dst_file = sftp.create(dst).await.map_err(|e| anyhow!("Failed to create dest: {}", e))?;
    let mut buffer = vec![0u8; 1024 * 1024]; 
    loop {
        let n = src_file.read(&mut buffer).await?;
        if n == 0 { break; }
        dst_file.write_all(&buffer[..n]).await?;
    }
    dst_file.flush().await?;
    dst_file.shutdown().await?;
    Ok(())
}

fn copy_recursive<'a>(sftp: &'a SftpSession, src: &'a str, dst: &'a str) -> Pin<Box<dyn Future<Output = Result<()>> + Send + 'a>> {
    Box::pin(async move {
        let _ = create_dir_all(sftp, dst).await; 
        let mut dir_stream = sftp.read_dir(src).await.map_err(|e| anyhow!("Failed to read dir {}: {}", src, e))?;
        while let Some(entry) = dir_stream.next() {
            let filename = entry.file_name();
            if filename == "." || filename == ".." { continue; }
            let sub_src = if src.ends_with('/') { format!("{}{}", src, filename) } else { format!("{}/{}", src, filename) };
            let sub_dst = if dst.ends_with('/') { format!("{}{}", dst, filename) } else { format!("{}/{}", dst, filename) };
            let meta = entry.metadata();
            if meta.is_dir() { 
                copy_recursive(sftp, &sub_src, &sub_dst).await?; 
            } else { 
                copy_file(sftp, &sub_src, &sub_dst).await?; 
            }
        }
        Ok(())
    })
}

// Optimized Fetch: Returns (Data, IsTruncated, TotalSize)
pub async fn fetch_file_smart(sftp: &SftpSession, remote_path: &str) -> Result<(Vec<u8>, bool, u64)> {
    let stat = sftp.metadata(remote_path).await
        .map_err(|e| anyhow!("File not found or inaccessible: {}", e))?;
    
    let size = stat.size.unwrap_or(0);
    
    if size > MAX_EDIT_SIZE_BYTES {
        // Partial Read (Preview)
        let mut remote_file = sftp.open(remote_path).await?;
        let mut buffer = vec![0u8; PREVIEW_SIZE_BYTES as usize];
        let n = remote_file.read(&mut buffer).await?;
        buffer.truncate(n);
        Ok((buffer, true, size))
    } else {
        // Full Read
        let mut remote_file = sftp.open(remote_path).await?;
        let mut buffer = Vec::with_capacity(size as usize);
        remote_file.read_to_end(&mut buffer).await?;
        Ok((buffer, false, size))
    }
}