use crate::connection::ssh_client::ClientHandler;
use crate::connection::ssh_session::HostConnection;
use russh::client::Handle;
use russh_sftp::client::SftpSession;
use russh_sftp::client::fs::DirEntry;
use std::path::Path;
use std::sync::Arc;
use anyhow::{Result, anyhow};
use std::sync::atomic::AtomicBool; 

pub mod fs;
pub mod transfer;

pub struct SftpManager {
    sftp: SftpSession,
    connection: Arc<HostConnection>,
}

impl SftpManager {
    pub async fn new(handle: &Handle<ClientHandler>, connection: Arc<HostConnection>) -> Result<Self> {
        let channel = handle.channel_open_session().await
            .map_err(|e| anyhow!("Failed to open SSH channel: {}", e))?;
        
        channel.request_subsystem(true, "sftp").await
            .map_err(|e| anyhow!("SFTP subsystem request failed: {}", e))?;

        let sftp = SftpSession::new(channel.into_stream()).await
            .map_err(|e| anyhow!("SFTP session init failed: {}", e))?;

        Ok(Self { sftp, connection })
    }

    pub fn session(&self) -> &SftpSession {
        &self.sftp
    }

    pub async fn create_dir_robust(&self, path: &str) -> Result<()> {
        if let Ok(_) = fs::create_dir_all(&self.sftp, path).await {
            return Ok(());
        }

        let cmd = format!("mkdir -p \"{}\"", path);
        match self.connection.exec_command(&cmd).await {
            Ok(_) => {
                if self.sftp.metadata(path).await.is_ok() {
                    Ok(())
                } else {
                    Err(anyhow!("Failed to create directory via SFTP and Shell fallback"))
                }
            },
            Err(e) => Err(anyhow!("Directory creation failed: {}", e))
        }
    }

    pub async fn read_dir(&self, path: &str) -> Result<Vec<DirEntry>> {
        fs::read_dir(&self.sftp, path).await
    }

    pub async fn create_empty(&self, path: &str) -> Result<()> {
        fs::create_empty(&self.sftp, path).await
    }

    pub async fn delete(&self, path: &str, recursive: bool) -> Result<()> {
        fs::delete(&self.sftp, path, recursive).await
    }

    pub async fn rename(&self, src: &str, dst: &str) -> Result<()> {
        fs::rename(&self.sftp, src, dst).await
    }

    pub async fn copy(&self, src: &str, dst: &str) -> Result<()> {
        fs::copy(&self.sftp, src, dst).await
    }

    // UPDATED: Return tuple for smart loading
    pub async fn fetch_for_edit(&self, remote_path: &str) -> Result<(Vec<u8>, bool, u64)> {
        fs::fetch_file_smart(&self.sftp, remote_path).await
    }

    pub async fn upload_file(&self, local_path: &Path, remote_path: &str) -> Result<()> {
        transfer::upload_file(self, local_path, remote_path).await
    }

    pub async fn upload_file_cancellable(
        &self, 
        local_path: &Path, 
        remote_path: &str, 
        cancel_token: Arc<AtomicBool>
    ) -> Result<()> {
        transfer::upload_file_cancellable(self, local_path, remote_path, cancel_token).await
    }

    pub async fn download_file(&self, remote_path: &str, local_path: &Path) -> Result<()> {
        transfer::download_file(&self.sftp, remote_path, local_path).await
    }
}