use russh_sftp::client::SftpSession;
use crate::connection::sftp::SftpManager;
use tokio::io::{AsyncReadExt, AsyncWriteExt};
use std::path::Path;
use anyhow::{Result, anyhow};
// Cleaned imports
use std::sync::{Arc, atomic::{AtomicBool, Ordering}};

pub async fn upload_file(
    manager: &SftpManager, 
    local_path: &Path, 
    remote_path: &str
) -> Result<()> {
    let dummy_token = Arc::new(AtomicBool::new(false));
    upload_file_cancellable(manager, local_path, remote_path, dummy_token).await
}

pub async fn upload_file_cancellable(
    manager: &SftpManager, 
    local_path: &Path, 
    remote_path: &str,
    cancel_token: Arc<AtomicBool>
) -> Result<()> {
    
    let local_bytes = tokio::fs::read(local_path).await
        .map_err(|e| anyhow!("Failed to read local: {}", e))?;
    
    if cancel_token.load(Ordering::Relaxed) { return Err(anyhow!("Cancelled by user")); }

    // 1. Ensure Parent Exists
    if let Some(parent_idx) = remote_path.rfind('/') {
        let parent = &remote_path[..parent_idx];
        if !parent.is_empty() {
            let _ = manager.create_dir_robust(parent).await;
        }
    }

    let temp_remote = format!("{}.part", remote_path);
    
    // Check cancel before creating remote file (Optimization)
    if cancel_token.load(Ordering::Relaxed) { return Err(anyhow!("Cancelled by user")); }

    // 2. Upload
    {
        let mut remote_file = manager.session().create(&temp_remote).await
            .map_err(|e| anyhow!("Failed to create temp file {}: {}", temp_remote, e))?;
        
        let chunk_size = 32 * 1024; // Lowered to 32KB to check cancel flag more often
        let mut offset = 0;
        
        while offset < local_bytes.len() {
            if cancel_token.load(Ordering::Relaxed) {
                let _ = remote_file.shutdown().await;
                let _ = manager.session().remove_file(&temp_remote).await;
                return Err(anyhow!("Cancelled by user"));
            }

            let end = std::cmp::min(offset + chunk_size, local_bytes.len());
            remote_file.write_all(&local_bytes[offset..end]).await?;
            offset = end;
        }

        remote_file.flush().await?;
        remote_file.shutdown().await?;
    }

    // 3. Verify
    let meta = manager.session().metadata(&temp_remote).await?;
    if meta.size.unwrap_or(0) != local_bytes.len() as u64 {
            let _ = manager.session().remove_file(&temp_remote).await;
            return Err(anyhow!("INTEGRITY FAILURE: Size mismatch"));
    }

    // 4. Move to Final
    let backup_remote = format!("{}.bak", remote_path);
    let _ = manager.session().remove_file(&backup_remote).await; 
    
    if manager.session().metadata(remote_path).await.is_ok() {
        let _ = manager.session().rename(remote_path, &backup_remote).await;
    }

    match manager.session().rename(&temp_remote, remote_path).await {
        Ok(_) => { 
            let _ = manager.session().remove_file(&backup_remote).await; 
            Ok(()) 
        },
        Err(e) => { 
            let _ = manager.session().rename(&backup_remote, remote_path).await; 
            Err(anyhow!("Rename failed (rolled back): {}", e)) 
        }
    }
}

pub async fn download_file(sftp: &SftpSession, remote_path: &str, local_path: &Path) -> Result<()> {
    let mut remote_file = sftp.open(remote_path).await
        .map_err(|e| anyhow!("Failed to open remote file: {}", e))?;
        
    let mut local_file = tokio::fs::File::create(local_path).await
        .map_err(|e| anyhow!("Failed to create local file: {}", e))?;
        
    let mut buffer = vec![0u8; 64 * 1024]; 
    loop {
        let n = remote_file.read(&mut buffer).await?;
        if n == 0 { break; }
        local_file.write_all(&buffer[..n]).await?;
    }
    
    local_file.flush().await?;
    Ok(())
}