// FILE: src/connection/ssh_session.rs
// -----------------------------------
// BATCH 30: Optimized "Hybrid" Order (Modern -> Legacy -> Fallback)

use crate::connection::ssh_client::ClientHandler;
use crate::config::models::HostConfig;
use russh::client::{Config, Handle};
use russh::keys::load_secret_key;
use ssh_key::{Algorithm, HashAlg, EcdsaCurve};
use std::sync::Arc;
use std::time::Duration;
use anyhow::{Result, Context, anyhow};

pub struct HostConnection {
    pub handle: Arc<Handle<ClientHandler>>,
    #[allow(dead_code)]
    pub host_id: uuid::Uuid,
}

impl HostConnection {
    pub async fn connect(host_config: &HostConfig) -> Result<Self> {
        println!("[SSH] Connecting to {}:{}", host_config.hostname, host_config.port);
        
        let mut config = Config::default();
        config.inactivity_timeout = Some(Duration::from_secs(120));
        config.keepalive_interval = Some(Duration::from_secs(10));
        config.window_size = 16 * 1024 * 1024;
        config.maximum_packet_size = 65535;

        // --- HYBRID COMPATIBILITY ORDER ---
        config.preferred.key = vec![
            // 1. Modern Standards (Preferred for Linux/OpenSSH)
            Algorithm::Ed25519,
            Algorithm::Ecdsa { curve: EcdsaCurve::NistP256 },
            Algorithm::Ecdsa { curve: EcdsaCurve::NistP384 },
            Algorithm::Ecdsa { curve: EcdsaCurve::NistP521 },

            // 2. Legacy RSA (SHA1)
            // Critical for Mikrotik/Cisco which advertise SHA2 but implement it in unexpected way.
            // Modern Linux (RSA-only) will skip this if they don't offer ssh-rsa.
            Algorithm::Rsa { hash: None }, 

            // 3. Legacy DSA (RouterOS / Old Switchgear)
            Algorithm::Dsa,

            // 4. Modern RSA (SHA2)
            // Fallback for strict Modern Linux servers that ONLY have RSA keys.
            Algorithm::Rsa { hash: Some(HashAlg::Sha512) }, 
            Algorithm::Rsa { hash: Some(HashAlg::Sha256) }, 
        ].into();

        let config_arc = Arc::new(config);

        let handler = ClientHandler::new();
        let addr = format!("{}:{}", host_config.hostname, host_config.port);
        
        let mut handle = tokio::time::timeout(
            Duration::from_secs(5),
            russh::client::connect(config_arc, addr, handler)
        ).await.context("TCP/SSH Connect timed out")??;

        println!("[SSH] Handshake complete. Authenticating user '{}'...", host_config.username);

        let auth_res = if let Some(key_path) = &host_config.key_path {
            let key_pair = load_secret_key(key_path, None)?;
            handle.authenticate_publickey(&host_config.username, Arc::new(key_pair)).await
        } else if let Some(password) = &host_config.password {
            handle.authenticate_password(&host_config.username, password).await
        } else {
            return Err(anyhow!("No credentials provided"));
        };

        if !auth_res? {
            return Err(anyhow!("Authentication failed"));
        }

        println!("[SSH] Authentication successful.");

        Ok(Self {
            handle: Arc::new(handle),
            host_id: host_config.id,
        })
    }

    pub async fn open_shell_channel(&self) -> Result<russh::Channel<russh::client::Msg>> {
        let channel = self.handle.channel_open_session().await?;
        channel.request_pty(false, "xterm-256color", 80, 24, 0, 0, &[]).await?;
        let _ = channel.window_change(80, 24, 0, 0).await;
        channel.request_shell(true).await?;
        Ok(channel)
    }

    pub async fn exec_command(&self, cmd: &str) -> Result<Vec<u8>> {
        let mut channel = self.handle.channel_open_session().await?;
        channel.exec(true, cmd).await?;
        let mut output = Vec::new();
        while let Some(msg) = channel.wait().await {
            match msg {
                russh::ChannelMsg::Data { ref data } => output.extend_from_slice(data),
                russh::ChannelMsg::ExitStatus { .. } => break,
                _ => {}
            }
        }
        Ok(output)
    }
}