use std::mem::ManuallyDrop;
use std::sync::atomic::{AtomicUsize, Ordering};
use std::os::windows::ffi::OsStrExt; // Fix for encode_wide

use windows::core::{implement, Result, Error, HRESULT}; // Removed Interface, c_void
use windows::Win32::Foundation::{BOOL, S_FALSE, S_OK, DV_E_FORMATETC, E_NOTIMPL, E_OUTOFMEMORY};
use windows::Win32::System::Com::{
    IDataObject, IDataObject_Impl, 
    FORMATETC, STGMEDIUM, STGMEDIUM_0, TYMED_HGLOBAL, 
    IEnumFORMATETC, IEnumFORMATETC_Impl, IAdviseSink, IEnumSTATDATA
};
use windows::Win32::System::Memory::{GlobalAlloc, GlobalLock, GlobalUnlock, GHND, GlobalSize};
use windows::Win32::System::Ole::CF_HDROP;
use windows::Win32::UI::Shell::DROPFILES;

#[implement(IDataObject)]
pub struct DataObject {
    stg_medium: STGMEDIUM,
}

impl DataObject {
    pub fn new(path: &std::path::Path) -> Result<Self> {
        let wide_path: Vec<u16> = path.as_os_str().encode_wide()
            .chain(std::iter::once(0))
            .chain(std::iter::once(0)) 
            .collect();
            
        let dropfiles_size = std::mem::size_of::<DROPFILES>();
        let path_size = wide_path.len() * 2;
        let total_size = dropfiles_size + path_size; 

        unsafe {
            let h_global = GlobalAlloc(GHND, total_size)?;
            let ptr = GlobalLock(h_global);
            
            if ptr.is_null() {
                return Err(Error::from(E_OUTOFMEMORY));
            }

            let drop_files = ptr as *mut DROPFILES;
            (*drop_files).pFiles = dropfiles_size as u32; 
            (*drop_files).fWide = BOOL(1); 

            let byte_ptr = ptr as *mut u8;
            let files_ptr = byte_ptr.add(dropfiles_size) as *mut u16;
            
            std::ptr::copy_nonoverlapping(wide_path.as_ptr(), files_ptr, wide_path.len());

            let _ = GlobalUnlock(h_global);

            let stg = STGMEDIUM {
                tymed: TYMED_HGLOBAL.0 as u32,
                u: STGMEDIUM_0 { hGlobal: h_global },
                pUnkForRelease: ManuallyDrop::new(None),
            };

            Ok(Self { stg_medium: stg })
        }
    }
}

impl IDataObject_Impl for DataObject {
    fn GetData(&self, format: *const FORMATETC) -> Result<STGMEDIUM> {
        unsafe {
            if (*format).cfFormat != CF_HDROP.0 {
                return Err(Error::from(DV_E_FORMATETC));
            }
            if ((*format).tymed & TYMED_HGLOBAL.0 as u32) == 0 {
                 return Err(Error::from(DV_E_FORMATETC));
            }

            let src_hglobal = self.stg_medium.u.hGlobal;
            let size = GlobalSize(src_hglobal);
            let dst_hglobal = GlobalAlloc(GHND, size)?;
            
            let src_ptr = GlobalLock(src_hglobal);
            let dst_ptr = GlobalLock(dst_hglobal);
            
            if src_ptr.is_null() || dst_ptr.is_null() {
                 return Err(Error::from(E_OUTOFMEMORY));
            }

            std::ptr::copy_nonoverlapping(src_ptr, dst_ptr, size);
            
            let _ = GlobalUnlock(src_hglobal);
            let _ = GlobalUnlock(dst_hglobal);

            Ok(STGMEDIUM {
                tymed: TYMED_HGLOBAL.0 as u32,
                u: STGMEDIUM_0 { hGlobal: dst_hglobal },
                pUnkForRelease: ManuallyDrop::new(None),
            })
        }
    }

    fn GetDataHere(&self, _pformatetc: *const FORMATETC, _pmedium: *mut STGMEDIUM) -> Result<()> {
        Err(Error::from(E_NOTIMPL))
    }

    fn QueryGetData(&self, format: *const FORMATETC) -> HRESULT {
        unsafe {
            if (*format).cfFormat == CF_HDROP.0 && ((*format).tymed & TYMED_HGLOBAL.0 as u32) != 0 {
                return S_OK;
            }
        }
        DV_E_FORMATETC
    }

    fn GetCanonicalFormatEtc(&self, _pformatectin: *const FORMATETC, _pformatetcout: *mut FORMATETC) -> HRESULT {
        E_NOTIMPL
    }

    fn SetData(&self, _pformatetc: *const FORMATETC, _pmedium: *const STGMEDIUM, _frelease: BOOL) -> Result<()> {
        Err(Error::from(E_NOTIMPL))
    }

    fn EnumFormatEtc(&self, _dwdirection: u32) -> Result<IEnumFORMATETC> {
        Ok(FormatEnumerator::new(0).into())
    }

    fn DAdvise(&self, _pformatetc: *const FORMATETC, _advf: u32, _padvsink: Option<&IAdviseSink>) -> Result<u32> {
        Err(Error::from(E_NOTIMPL))
    }

    fn DUnadvise(&self, _dwconnection: u32) -> Result<()> {
        Err(Error::from(E_NOTIMPL))
    }

    fn EnumDAdvise(&self) -> Result<IEnumSTATDATA> {
        Err(Error::from(E_NOTIMPL))
    }
}

#[implement(IEnumFORMATETC)]
pub struct FormatEnumerator {
    index: AtomicUsize,
}

impl FormatEnumerator {
    pub fn new(start: usize) -> Self {
        Self { index: AtomicUsize::new(start) }
    }
}

impl IEnumFORMATETC_Impl for FormatEnumerator {
    fn Next(&self, celt: u32, rgelt: *mut FORMATETC, pceltfetched: *mut u32) -> Result<()> {
        let idx = self.index.load(Ordering::Relaxed);
        
        if idx == 0 && celt > 0 {
            unsafe {
                let format = FORMATETC {
                    cfFormat: CF_HDROP.0,
                    ptd: std::ptr::null_mut(),
                    dwAspect: 1, 
                    lindex: -1,
                    tymed: TYMED_HGLOBAL.0 as u32,
                };
                
                *rgelt = format;
                
                if !pceltfetched.is_null() {
                    *pceltfetched = 1;
                }
                
                self.index.store(1, Ordering::Relaxed);
                return Ok(());
            }
        }
        
        if !pceltfetched.is_null() {
            unsafe { *pceltfetched = 0; }
        }
        Err(Error::from(S_FALSE)) 
    }

    fn Skip(&self, celt: u32) -> Result<()> {
        let idx = self.index.load(Ordering::Relaxed);
        self.index.store(idx + celt as usize, Ordering::Relaxed);
        Ok(())
    }

    fn Reset(&self) -> Result<()> {
        self.index.store(0, Ordering::Relaxed);
        Ok(())
    }

    fn Clone(&self) -> Result<IEnumFORMATETC> {
        let idx = self.index.load(Ordering::Relaxed);
        Ok(FormatEnumerator::new(idx).into())
    }
}