pub mod data;
pub mod source;

use std::path::Path;
use std::os::windows::ffi::OsStrExt;
use windows::core::{Result, Error};
use windows::Win32::Foundation::{E_FAIL, HWND, BOOL, HANDLE, GlobalFree};
use windows::Win32::System::Memory::{GlobalAlloc, GlobalLock, GlobalUnlock, GHND};
use windows::Win32::UI::Shell::DROPFILES;
use windows::Win32::System::DataExchange::{OpenClipboard, EmptyClipboard, SetClipboardData, CloseClipboard};

pub fn copy_to_clipboard(local_path: &Path) -> Result<()> {
    unsafe {
        let abs_path = std::fs::canonicalize(local_path).map_err(|_| Error::from(E_FAIL))?;
        let abs_path_str = abs_path.to_string_lossy();
        
        let clean_path = if abs_path_str.starts_with(r"\\?\") {
            Path::new(&abs_path_str[4..])
        } else {
            abs_path.as_path()
        };

        let wide_path: Vec<u16> = clean_path.as_os_str().encode_wide()
            .chain(std::iter::once(0))
            .chain(std::iter::once(0)) 
            .collect();
            
        let dropfiles_size = std::mem::size_of::<DROPFILES>();
        let path_size = wide_path.len() * 2;
        let total_size = dropfiles_size + path_size; 

        let h_global = GlobalAlloc(GHND, total_size)?;
        let ptr = GlobalLock(h_global);
        
        if ptr.is_null() {
            return Err(Error::from(E_FAIL));
        }

        let drop_files = ptr as *mut DROPFILES;
        (*drop_files).pFiles = dropfiles_size as u32; 
        (*drop_files).fWide = BOOL(1); 

        let byte_ptr = ptr as *mut u8;
        let files_ptr = byte_ptr.add(dropfiles_size) as *mut u16;
        std::ptr::copy_nonoverlapping(wide_path.as_ptr(), files_ptr, wide_path.len());

        // FIXED: Handle Result
        let _ = GlobalUnlock(h_global);

        let mut success = false;
        for _ in 0..5 {
            if OpenClipboard(HWND(0)).is_ok() {
                let _ = EmptyClipboard();
                if SetClipboardData(15, HANDLE(h_global.0 as isize)).is_ok() {
                    success = true;
                }
                let _ = CloseClipboard();
                break;
            }
            std::thread::sleep(std::time::Duration::from_millis(50));
        }

        if !success {
            let _ = GlobalFree(h_global);
            return Err(Error::from(E_FAIL));
        }
    }

    Ok(())
}