use std::sync::atomic::{AtomicBool, Ordering};
use std::thread;
use std::time::Duration;
use windows::Win32::Foundation::{HINSTANCE, HWND, LPARAM, LRESULT, WPARAM};
use windows::Win32::System::Threading::GetCurrentThreadId;
use windows::Win32::UI::WindowsAndMessaging::{
    CallNextHookEx, GetClassNameW, SetWindowsHookExW, UnhookWindowsHookEx,
    WindowFromPoint, GetCursorPos, SetForegroundWindow,
    WH_MOUSE_LL, WM_LBUTTONDOWN, MSG, HC_ACTION, GetMessageW, 
    PostThreadMessageW, TranslateMessage, DispatchMessageW, WM_QUIT, HHOOK
};
use windows::Win32::UI::Input::KeyboardAndMouse::{
    SendInput, INPUT, INPUT_0, INPUT_KEYBOARD, KEYBDINPUT, 
    VK_CONTROL, VK_V, VK_MENU, KEYEVENTF_KEYUP, GetAsyncKeyState, KEYBD_EVENT_FLAGS
};

static IS_HOOK_ACTIVE: AtomicBool = AtomicBool::new(false);
static mut H_HOOK: HHOOK = HHOOK(0);
static mut THREAD_ID: u32 = 0;

// Batch 1: RAII Guard to ensure hook is always removed
struct HookGuard(HHOOK);

impl Drop for HookGuard {
    fn drop(&mut self) {
        if !self.0.is_invalid() {
            unsafe {
                let _ = UnhookWindowsHookEx(self.0);
                H_HOOK = HHOOK(0);
            }
            IS_HOOK_ACTIVE.store(false, Ordering::Relaxed);
            // log::debug!("[HOOK] Cleaned up via Drop.");
        }
    }
}

pub fn enable_paste_hook() {
    if IS_HOOK_ACTIVE.load(Ordering::Relaxed) {
        // log::debug!("[HOOK] Already active.");
        return;
    }

    IS_HOOK_ACTIVE.store(true, Ordering::Relaxed);

    thread::spawn(|| {
        unsafe {
            THREAD_ID = GetCurrentThreadId();
            
            let hook_res = SetWindowsHookExW(
                WH_MOUSE_LL,
                Some(mouse_hook_proc),
                HINSTANCE(0),
                0
            );

            if let Ok(h) = hook_res {
                H_HOOK = h;
                // Create RAII Guard. 
                // When this function exits (loops ends or panic), this guard drops and unhooks.
                let _guard = HookGuard(h); 
                
                // log::debug!("[HOOK] Installed.");

                // Watchdog timer to kill hook if user doesn't click
                let _timer = thread::spawn(|| {
                    thread::sleep(Duration::from_secs(15));
                    if IS_HOOK_ACTIVE.load(Ordering::Relaxed) {
                        // log::debug!("[HOOK] Timeout.");
                        IS_HOOK_ACTIVE.store(false, Ordering::Relaxed);
                        let tid = THREAD_ID;
                        if tid != 0 {
                            let _ = PostThreadMessageW(tid, WM_QUIT, WPARAM(0), LPARAM(0));
                        }
                    }
                });

                let mut msg = MSG::default();
                // Message Loop
                while GetMessageW(&mut msg, HWND(0), 0, 0).as_bool() {
                    // Check atomic flag in case it was toggled from outside the message loop
                    if !IS_HOOK_ACTIVE.load(Ordering::Relaxed) { break; }
                    TranslateMessage(&msg);
                    DispatchMessageW(&msg);
                }
                
                // _guard drops here automatically
            } else {
                eprintln!("[HOOK] Failed to install hook.");
                IS_HOOK_ACTIVE.store(false, Ordering::Relaxed);
            }
        }
    });
}

unsafe extern "system" fn mouse_hook_proc(n_code: i32, w_param: WPARAM, l_param: LPARAM) -> LRESULT {
    if n_code == HC_ACTION as i32 && w_param.0 == WM_LBUTTONDOWN as usize {
        // Check if Alt is held down (0x8000 is the 'key down' bit)
        if (GetAsyncKeyState(VK_MENU.0 as i32) as u16 & 0x8000) != 0 {
            let mut point = windows::Win32::Foundation::POINT::default();
            let _ = GetCursorPos(&mut point);
            let hwnd = WindowFromPoint(point);
            
            if is_explorer_window(hwnd) {
                SetForegroundWindow(hwnd);
                send_paste_sequence();
                
                // Signal loop to exit
                IS_HOOK_ACTIVE.store(false, Ordering::Relaxed);
                let tid = THREAD_ID;
                if tid != 0 {
                    let _ = PostThreadMessageW(tid, WM_QUIT, WPARAM(0), LPARAM(0));
                }
                
                // Swallow the click so we don't select the file underneath immediately
                return LRESULT(1); 
            }
        }
    }
    CallNextHookEx(H_HOOK, n_code, w_param, l_param)
}

unsafe fn is_explorer_window(hwnd: HWND) -> bool {
    let mut current = hwnd;
    for _ in 0..8 {
        if current.0 == 0 { break; }
        
        let mut buffer = [0u16; 256];
        let len = GetClassNameW(current, &mut buffer);
        let name = String::from_utf16_lossy(&buffer[..len as usize]);
        
        // Common Explorer class names
        if name == "CabinetWClass" || name == "ExploreWClass" || name == "Progman" || name == "WorkerW" {
            return true;
        }
        
        current = windows::Win32::UI::WindowsAndMessaging::GetParent(current);
    }
    false
}

unsafe fn send_paste_sequence() {
    let inputs = [
        input_key(VK_MENU, true),    // Alt Up
        input_key(VK_CONTROL, false),// Ctrl Down
        input_key(VK_V, false),      // V Down
        input_key(VK_V, true),       // V Up
        input_key(VK_CONTROL, true), // Ctrl Up
        input_key(VK_MENU, false),   // Alt Down (Reset state if user still holding it? mostly harmless)
    ];
    
    SendInput(&inputs, std::mem::size_of::<INPUT>() as i32);
}

unsafe fn input_key(vk: windows::Win32::UI::Input::KeyboardAndMouse::VIRTUAL_KEY, up: bool) -> INPUT {
    let flags = if up { KEYEVENTF_KEYUP } else { KEYBD_EVENT_FLAGS(0) };
    INPUT {
        r#type: INPUT_KEYBOARD,
        Anonymous: INPUT_0 {
            ki: KEYBDINPUT {
                wVk: vk,
                wScan: 0,
                dwFlags: flags,
                time: 0,
                dwExtraInfo: 0,
            }
        }
    }
}