// FILE: src/main.rs
// -----------------

#![cfg_attr(not(debug_assertions), windows_subsystem = "windows")]

mod constants;
mod config;
mod connection;
mod serial;
mod monitor;
mod commands;
mod interop;
mod terminal;

use commands::app_cmds::AppState;
use std::sync::Arc;
use tokio::sync::RwLock;
use tauri::{Emitter, Manager};

fn main() {
    if std::env::var("RUST_LOG").is_err() {
        std::env::set_var("RUST_LOG", "info");
    }
    env_logger::init();
    
    log::info!("Starting OmniTerminal...");

    let config_store = config::store::ConfigStore::load();
    let ssh_manager = connection::manager::ConnectionManager::new();
    
    let serial_sessions = Arc::new(RwLock::new(std::collections::HashMap::new()));
    let ssh_sessions = Arc::new(RwLock::new(std::collections::HashMap::new()));
    let sftp_sessions = Arc::new(RwLock::new(std::collections::HashMap::new()));
    
    let serial_read_buffers = Arc::new(RwLock::new(std::collections::HashMap::new()));
    let serial_status = Arc::new(RwLock::new(std::collections::HashMap::new()));
    let sftp_cancel_tokens = Arc::new(RwLock::new(std::collections::HashMap::new()));
    let local_ptys = Arc::new(RwLock::new(std::collections::HashMap::new()));

    let app_state = AppState {
        config_store,
        ssh_manager,
        serial_sessions,
        ssh_sessions,
        sftp_sessions,
        serial_read_buffers,
        serial_status,
        sftp_cancel_tokens,
        local_ptys,
    };

    tauri::Builder::default()
        .plugin(tauri_plugin_single_instance::init(|app, _args, _cwd| {
            let _ = app.get_webview_window("main")
                       .expect("no main window")
                       .set_focus();
        }))
        .plugin(tauri_plugin_shell::init())
        .plugin(tauri_plugin_fs::init())
        .plugin(tauri_plugin_dialog::init())
        .manage(app_state)
        .on_window_event(|window, event| {
            if let tauri::WindowEvent::CloseRequested { api, .. } = event {
                api.prevent_close();
                if let Err(e) = window.emit("app-close-request", ()) {
                    eprintln!("Failed to emit close request: {}", e);
                    std::process::exit(0);
                }
            }
        })
        .invoke_handler(tauri::generate_handler![
            // Security
            commands::app_cmds::get_lock_status,
            commands::app_cmds::get_security_status,
            commands::app_cmds::unlock_app,
            commands::app_cmds::set_master_password,

            // App & Config
            commands::app_cmds::get_app_config,
            commands::app_cmds::save_app_config,
            commands::app_cmds::save_ssh_host,
            commands::app_cmds::delete_ssh_host,
            commands::app_cmds::get_inventory_tree,
            commands::app_cmds::force_app_exit,
            commands::app_cmds::save_serial_device, // <--- REGISTERED HERE
            
            // Tools
            commands::tool_cmds::check_local_tools,

            // SSH
            commands::ssh_cmds::connect_ssh,
            commands::ssh_cmds::send_ssh_data, 
            commands::ssh_cmds::poll_ssh,
            commands::ssh_cmds::disconnect_ssh,
            commands::ssh_cmds::resize_ssh_session,
            commands::ssh_cmds::exec_ssh_silent,
            
            // Serial
            commands::serial_cmds::list_serial_devices,
            commands::serial_cmds::start_serial_session,
            commands::serial_cmds::send_serial_data,
            commands::serial_cmds::poll_serial,
            commands::serial_cmds::toggle_serial_pin,
            
            // SFTP
            commands::sftp::session::sftp_open_session,
            commands::sftp::session::sftp_close_session,
            commands::sftp::io::sftp_list_files,
            commands::sftp::io::sftp_read_file,
            commands::sftp::io::sftp_write_file,
            commands::sftp::io::sftp_write_binary, 
            commands::sftp::io::sftp_create_entry,
            commands::sftp::io::sftp_delete,
            commands::sftp::io::sftp_rename,
            commands::sftp::io::sftp_copy,
            commands::sftp::io::sftp_download,
            commands::sftp::io::sftp_to_clipboard, 
            commands::app_cmds::cancel_sftp_transfer,

            // SYNC
            commands::sftp::sync::sftp_sync_dry_run,
            commands::sftp::sync::sftp_sync_execute,

            // Local PTY
            commands::local_term::spawn_local_pty,
            commands::local_term::send_local_pty_data,
            commands::local_term::resize_local_pty,
        ])
        .run(tauri::generate_context!())
        .expect("error while running tauri application");
}