use crate::connection::ssh_session::HostConnection;
use crate::monitor::models::{HostStats, MonitorState};
use std::sync::Arc;
use tokio::time::{sleep, Duration};
use log::warn;
use tauri::{Window, Emitter};

const LINUX_POLL_CMD: &str = "export LC_ALL=C; cat /proc/stat /proc/meminfo /proc/net/dev; echo '---'; df -P";

#[derive(serde::Serialize, Clone)]
struct MetricPayload {
    id: String,
    stats: HostStats,
}

pub struct MetricsEngine {
    session_id: String, 
    connection: Arc<HostConnection>,
    window: Window, 
    state: MonitorState,
}

impl MetricsEngine {
    pub fn new(session_id: String, connection: Arc<HostConnection>, window: Window) -> Self {
        Self {
            session_id,
            connection,
            window,
            state: MonitorState::default(),
        }
    }

    pub async fn run_loop(mut self) {
        loop {
            // REMOVED: self.window.is_destroying() check.
            // We rely on .emit() returning an error to detect closure.

            match self.connection.exec_command(LINUX_POLL_CMD).await {
                Ok(output) => {
                    let output_str = String::from_utf8_lossy(&output);
                    
                    if output_str.trim().is_empty() {
                         sleep(Duration::from_secs(2)).await;
                         continue;
                    }

                    let stats = self.parse_metrics(&output_str);
                    
                    let payload = MetricPayload {
                        id: self.session_id.clone(),
                        stats,
                    };

                    // The actual check for window validity happens here
                    if let Err(e) = self.window.emit("ssh-metrics", payload) {
                        warn!("Metrics emission failed (window closed?): {}", e);
                        break;
                    }
                }
                Err(e) => {
                    warn!("Metrics poll error: {}", e);
                    sleep(Duration::from_secs(5)).await;
                }
            }
            sleep(Duration::from_secs(2)).await;
        }
    }

    fn parse_metrics(&mut self, raw: &str) -> HostStats {
        let mut stats = HostStats::default();
        let mut mem_total = 0;
        let mut mem_available = 0;
        
        let sections: Vec<&str> = raw.split("\n---\n").collect();
        let main_section = sections.get(0).unwrap_or(&raw);
        let disk_section = sections.get(1);

        // Accumulators
        let mut tick_rx = 0;
        let mut tick_tx = 0;
        let mut max_core_usage = 0.0;

        for line in main_section.lines() {
            let line = line.trim();
            if line.is_empty() { continue; }
            let parts: Vec<&str> = line.split_whitespace().collect();
            if parts.is_empty() { continue; }

            // 1. CPU
            if parts[0].starts_with("cpu") {
                if parts.len() >= 5 {
                    let vals: Vec<u64> = parts.iter().skip(1).filter_map(|s| s.parse().ok()).collect();
                    if vals.len() >= 4 {
                        let work = vals[0] + vals[1] + vals[2]; 
                        let idle = vals[3];
                        let total = work + idle;

                        if parts[0] == "cpu" {
                            let diff_total = total.saturating_sub(self.state.last_cpu_total);
                            let diff_work = work.saturating_sub(self.state.last_cpu_work);
                            if diff_total > 0 {
                                stats.cpu_usage = ((diff_work as f64 / diff_total as f64) * 100.0) as f32;
                            }
                            self.state.last_cpu_total = total;
                            self.state.last_cpu_work = work;
                        } else {
                            let core_id = parts[0].to_string();
                            let (prev_work, prev_total) = *self.state.core_states.get(&core_id).unwrap_or(&(0, 0));
                            let diff_total = total.saturating_sub(prev_total);
                            let diff_work = work.saturating_sub(prev_work);
                            if diff_total > 0 {
                                let usage = ((diff_work as f64 / diff_total as f64) * 100.0) as f32;
                                if usage > max_core_usage { max_core_usage = usage; }
                            }
                            self.state.core_states.insert(core_id, (work, total));
                        }
                    }
                }
            }
            // 2. RAM
            else if line.starts_with("MemTotal:") { mem_total = parse_kb(line) * 1024; }
            else if line.starts_with("MemAvailable:") { mem_available = parse_kb(line) * 1024; }
            
            // 3. NET
            else if line.contains(':') {
                let col_parts: Vec<&str> = line.split(':').collect();
                if col_parts.len() >= 2 {
                    let iface = col_parts[0].trim();
                    let stats_str = col_parts[1].trim();
                    
                    if !(iface == "lo" || iface.starts_with("veth") || iface.starts_with("docker") || iface.starts_with("br-")) {
                        let nums: Vec<u64> = stats_str.split_whitespace().filter_map(|s| s.parse().ok()).collect();
                        if nums.len() >= 9 {
                            tick_rx += nums[0];
                            tick_tx += nums[8];
                        }
                    }
                }
            }
        }

        stats.highest_cpu_core_usage = max_core_usage;

        // Net Speed
        let now = std::time::Instant::now();
        let delta = now.duration_since(self.state.last_check_time).as_secs_f32();
        if delta > 0.1 && self.state.last_net_rx > 0 {
            if tick_rx >= self.state.last_net_rx {
                let diff_rx = tick_rx - self.state.last_net_rx;
                stats.net_rx_mbps = (diff_rx as f32 * 8.0) / (1_000_000.0 * delta);
            }
            if tick_tx >= self.state.last_net_tx {
                let diff_tx = tick_tx - self.state.last_net_tx;
                stats.net_tx_mbps = (diff_tx as f32 * 8.0) / (1_000_000.0 * delta);
            }
        }
        self.state.last_net_rx = tick_rx;
        self.state.last_net_tx = tick_tx;
        self.state.last_check_time = now;

        // RAM
        stats.ram_total = mem_total;
        if mem_total > 0 && mem_available <= mem_total {
            stats.ram_used = mem_total - mem_available;
        }

        // 4. Disk
        if let Some(ds) = disk_section {
            for line in ds.lines().skip(1) { 
                let parts: Vec<&str> = line.split_whitespace().collect();
                if parts.len() >= 6 {
                    let cap_str = parts[4].trim_end_matches('%');
                    let mount_point = parts[5];
                    
                    if let Ok(cap) = cap_str.parse::<f32>() {
                        stats.disk_details.push((mount_point.to_string(), cap));
                        if mount_point == "/" {
                            stats.root_disk_usage = cap;
                        }
                    }
                }
            }
        }

        stats
    }
}

fn parse_kb(line: &str) -> u64 {
    for part in line.split_whitespace() {
        if let Ok(val) = part.parse::<u64>() { return val; }
    }
    0
}