use crate::serial::models::DeviceFingerprint;
use windows::core::{PCWSTR, GUID};
use windows::Win32::Devices::DeviceAndDriverInstallation::{
    SetupDiDestroyDeviceInfoList, SetupDiEnumDeviceInfo, SetupDiGetClassDevsW,
    SetupDiGetDeviceRegistryPropertyW, DIGCF_PRESENT, DIGCF_DEVICEINTERFACE,
    SP_DEVINFO_DATA, SPDRP_FRIENDLYNAME, SPDRP_HARDWAREID, HDEVINFO,
};
use windows::Win32::Foundation::INVALID_HANDLE_VALUE;

const GUID_DEVINTERFACE_COMPORT: GUID = 
    GUID::from_u128(0x86e0d1e0_8089_11d0_9ce4_08003e301f73);

pub fn enumerate_devices() -> Vec<DeviceFingerprint> {
    let mut devices = Vec::new();

    unsafe {
        let h_dev_info = match SetupDiGetClassDevsW(
            Some(&GUID_DEVINTERFACE_COMPORT),
            PCWSTR::null(),
            None,
            DIGCF_PRESENT | DIGCF_DEVICEINTERFACE,
        ) {
            Ok(handle) => handle,
            Err(_) => return devices,
        };

        if h_dev_info.0 == INVALID_HANDLE_VALUE.0 {
             return devices;
        }

        let mut member_index = 0;
        loop {
            let mut dev_info_data = SP_DEVINFO_DATA {
                cbSize: std::mem::size_of::<SP_DEVINFO_DATA>() as u32,
                ..Default::default()
            };

            if let Err(_) = SetupDiEnumDeviceInfo(h_dev_info, member_index, &mut dev_info_data) {
                break;
            }

            let friendly_name = get_registry_property(h_dev_info, &mut dev_info_data, SPDRP_FRIENDLYNAME);
            let hardware_id = get_registry_property(h_dev_info, &mut dev_info_data, SPDRP_HARDWAREID);
            
            let port_name = extract_com_port(&friendly_name);
            // instance_id is derived from hardware_id here, but we suppress the warning by using it
            let _instance_id = hardware_id.clone(); 

            if let Some(port) = port_name {
                let effective_id = port.clone(); 

                devices.push(DeviceFingerprint {
                    port_name: port,
                    friendly_name: if friendly_name.is_empty() { "Unknown Device".to_string() } else { friendly_name },
                    hardware_id: hardware_id,
                    instance_id: effective_id,
                });
            }

            member_index += 1;
        }
        let _ = SetupDiDestroyDeviceInfoList(h_dev_info);
    }

    devices
}

unsafe fn get_registry_property(
    h_dev_info: HDEVINFO,
    dev_info_data: &mut SP_DEVINFO_DATA,
    property: u32, 
) -> String {
    let mut buffer = [0u8; 1024];
    let mut required_size = 0;

    let res = SetupDiGetDeviceRegistryPropertyW(
        h_dev_info,
        dev_info_data,
        property,
        None, 
        Some(&mut buffer),
        Some(&mut required_size),
    );

    if res.is_ok() {
        let len = (required_size as usize) / 2; 
        if len > 1 {
            let ptr = buffer.as_ptr() as *const u16;
            let slice = std::slice::from_raw_parts(ptr, len - 1); 
            return String::from_utf16_lossy(slice);
        }
    }
    "Unknown".to_string()
}

fn extract_com_port(friendly_name: &str) -> Option<String> {
    let start = friendly_name.rfind("(COM")?;
    let end = friendly_name.rfind(")")?;
    if start < end {
        return Some(friendly_name[start+1..end].to_string());
    }
    None
}