use serialport::SerialPort;
use std::sync::{Arc, Mutex};
use tokio::sync::mpsc;
use crate::serial::models::BytecodeConfig;
use std::io::{Read, Write}; 

#[derive(Clone)]
pub struct SerialStream {
    // We only need the Mutex for the Write handle.
    // The Read handle is moved entirely to the background thread.
    write_port: Arc<Mutex<Box<dyn SerialPort>>>,
    config: BytecodeConfig,
}

impl SerialStream {
    pub fn new(port: Box<dyn SerialPort>, config: BytecodeConfig) -> Self {
        Self {
            write_port: Arc::new(Mutex::new(port)),
            config,
        }
    }

    // UPDATED: Now takes an error_tx channel to notify Supervisor of death
    pub fn spawn_reader(&self, tx: mpsc::Sender<Vec<u8>>, error_tx: mpsc::Sender<()>) {
        // Create a native clone of the port handle for the reader thread.
        // This avoids locking the 'write_port' mutex during blocking reads.
        let mut read_port = {
            let p = self.write_port.lock().unwrap();
            match p.try_clone() {
                Ok(cloned) => cloned,
                Err(e) => {
                    log::error!("Failed to clone serial port for reading: {}", e);
                    let _ = error_tx.blocking_send(());
                    return;
                }
            }
        };

        tokio::task::spawn_blocking(move || {
            let mut buf = [0u8; 4096];
            loop {
                // Read directly from the cloned handle. No Mutex needed!
                match read_port.read(&mut buf) {
                    Ok(n) if n > 0 => {
                        let data = buf[0..n].to_vec();
                        if tx.blocking_send(data).is_err() {
                            break; // Channel closed
                        }
                    }
                    Ok(_) => {} // 0 bytes (timeout usually)
                    Err(ref e) if e.kind() == std::io::ErrorKind::TimedOut => {
                        // Timeout is fine, just loop
                        continue;
                    }
                    Err(e) => {
                        // CRITICAL FIX: Notify Supervisor immediately on OS Error 22 / Disconnect
                        log::warn!("Serial RX Error: {}", e);
                        let _ = error_tx.blocking_send(()); 
                        break; 
                    }
                }
            }
        });
    }

    pub async fn write_data(&self, data: &str) -> Result<(), String> {
        let mut payload = data.as_bytes().to_vec();

        if self.config.force_crlf {
            if !payload.ends_with(b"\r\n") {
                 payload.extend_from_slice(b"\r\n");
            }
        }

        let mut final_packet = self.config.tx_prefix.clone();
        final_packet.extend(payload);
        final_packet.extend(&self.config.tx_suffix);

        let port_clone = self.write_port.clone();
        
        // This lock is now uncontested by the reader!
        let res = tokio::task::spawn_blocking(move || {
            let mut p = port_clone.lock().unwrap();
            p.write_all(&final_packet).and_then(|_| p.flush())
        }).await;

        match res {
            Ok(Ok(_)) => Ok(()),
            Ok(Err(e)) => Err(e.to_string()),
            Err(e) => Err(e.to_string()), 
        }
    }

    // --- Hardware Control ---

    pub fn set_dtr(&self, level: bool) -> Result<(), String> {
        let mut p = self.write_port.lock().unwrap();
        p.write_data_terminal_ready(level).map_err(|e| e.to_string())
    }

    pub fn set_rts(&self, level: bool) -> Result<(), String> {
        let mut p = self.write_port.lock().unwrap();
        p.write_request_to_send(level).map_err(|e| e.to_string())
    }
}