use crate::serial::{enumerator, stream::SerialStream, models::BytecodeConfig, models::SerialCommand};
use tokio::sync::mpsc;
use tokio::time::{sleep, Duration};
use std::sync::{Arc, Mutex};
use log::{info, warn}; 

#[derive(Debug, Clone)]
pub enum ConnectionState {
    Connected,
    #[allow(dead_code)]
    Disconnected,
    Searching,
    Error(#[allow(dead_code)] String),
}

pub struct SerialSupervisor {
    target_instance_id: String, 
    baud_rate: u32,
    config: BytecodeConfig,
    state_tx: mpsc::Sender<ConnectionState>, 
    data_tx: mpsc::Sender<Vec<u8>>, 
    write_rx: mpsc::Receiver<SerialCommand>, 
    current_stream: Arc<Mutex<Option<SerialStream>>>,
}

impl SerialSupervisor {
    pub fn new(
        target_instance_id: String,
        baud_rate: u32,
        config: BytecodeConfig,
        state_tx: mpsc::Sender<ConnectionState>,
        data_tx: mpsc::Sender<Vec<u8>>,
        write_rx: mpsc::Receiver<SerialCommand>,
    ) -> Self {
        Self {
            target_instance_id,
            baud_rate,
            config,
            state_tx,
            data_tx,
            write_rx,
            current_stream: Arc::new(Mutex::new(None)),
        }
    }

    pub async fn run(mut self) {
        let _ = self.state_tx.send(ConnectionState::Searching).await;

        // Internal channel to detect when the Reader Thread dies (e.g. unplugged)
        let (err_tx, mut err_rx) = mpsc::channel(1);

        loop {
            let is_connected = { self.current_stream.lock().unwrap().is_some() };
            
            if !is_connected {
                // SCANNING PHASE
                if let Some(port_name) = self.find_port_by_instance().await {
                    match self.connect(&port_name) {
                        Ok(stream) => {
                            info!("Connected to {}", port_name);
                            let _ = self.state_tx.send(ConnectionState::Connected).await;
                            
                            // Pass the error transmitter to the stream
                            stream.spawn_reader(self.data_tx.clone(), err_tx.clone());
                            
                            *self.current_stream.lock().unwrap() = Some(stream);
                        }
                        Err(e) => {
                            let _ = self.state_tx.send(ConnectionState::Error(e.to_string())).await;
                            sleep(Duration::from_secs(2)).await;
                            continue;
                        }
                    }
                } else {
                    // While searching, we only listen for commands (to perhaps cancel?) or just sleep
                    tokio::select! {
                        _ = sleep(Duration::from_secs(1)) => {},
                        _ = self.write_rx.recv() => { /* Drop commands while disconnected */ } 
                    }
                    continue; 
                }
            }

            // CONNECTED PHASE
            tokio::select! {
                // 1. Handle commands from UI
                Some(cmd) = self.write_rx.recv() => {
                    let stream_to_use = {
                        let lock = self.current_stream.lock().unwrap();
                        lock.clone()
                    };
                    
                    if let Some(stream) = stream_to_use {
                        match cmd {
                            SerialCommand::WriteData(data) => {
                                if let Err(e) = stream.write_data(&data).await {
                                    warn!("Write failed: {}", e);
                                    // Let the error flow to err_rx or handle here
                                }
                            },
                            SerialCommand::SetDtr(state) => {
                                let _ = stream.set_dtr(state); 
                            },
                            SerialCommand::SetRts(state) => {
                                let _ = stream.set_rts(state);
                            }
                        }
                    }
                }
                
                // 2. Handle Stream Death (Instant feedback)
                Some(_) = err_rx.recv() => {
                    warn!("Stream died (hardware disconnect). Re-entering search mode.");
                    let _ = self.state_tx.send(ConnectionState::Searching).await;
                    *self.current_stream.lock().unwrap() = None;
                }

                // 3. Periodic sanity check (Backup for driver hangs)
                _ = sleep(Duration::from_secs(1)) => {
                    if self.find_port_by_instance().await.is_none() {
                        // Driver finally updated
                        if self.current_stream.lock().unwrap().is_some() {
                             warn!("Device vanished from registry. Re-entering search mode.");
                             let _ = self.state_tx.send(ConnectionState::Searching).await;
                             *self.current_stream.lock().unwrap() = None;
                        }
                    }
                }
            }
        }
    }

    async fn find_port_by_instance(&self) -> Option<String> {
        let target = self.target_instance_id.clone();
        tokio::task::spawn_blocking(move || {
            let devices = enumerator::enumerate_devices();
            for dev in devices {
                // Loose matching to handle varying Windows PnP IDs
                if dev.instance_id == target 
                   || dev.port_name == target 
                   || dev.port_name.contains(&target) 
                   || target.contains(&dev.port_name) {
                    return Some(dev.port_name);
                }
            }
            None
        }).await.unwrap_or(None)
    }

    fn connect(&self, port_name: &str) -> Result<SerialStream, serialport::Error> {
        let port = serialport::new(port_name, self.baud_rate)
            .timeout(Duration::from_millis(100))
            .open()?;
        Ok(SerialStream::new(port, self.config.clone()))
    }
}