use portable_pty::{CommandBuilder, NativePtySystem, PtySize, PtySystem, MasterPty};
use std::sync::{Arc, Mutex};
use std::io::{Read, Write};
use anyhow::{Result, anyhow};
use tauri::{Emitter, Window};

pub struct PtySession {
    master: Arc<Mutex<Box<dyn MasterPty + Send>>>,
    // We keep the writer separate to allow async writes without locking the whole structure constantly
    writer: Arc<Mutex<Box<dyn Write + Send>>>,
}

#[derive(serde::Serialize, Clone)]
struct PtyDataPayload {
    id: String,
    data: Vec<u8>,
}

impl PtySession {
    pub fn spawn(session_id: String, window: Window, rows: u16, cols: u16) -> Result<Self> {
        let pty_system = NativePtySystem::default();

        let pair = pty_system.openpty(PtySize {
            rows,
            cols,
            pixel_width: 0,
            pixel_height: 0,
        })?;

        // Determine shell based on OS (Assuming Windows for this specific request, but good to be generic)
        let cmd = if cfg!(target_os = "windows") {
            CommandBuilder::new("powershell.exe")
        } else {
            CommandBuilder::new("bash")
        };

        let _child = pair.slave.spawn_command(cmd)?;

        // Reader Thread
        // We clone the reader handle and move it to a dedicated thread that pushes events to Tauri
        let mut reader = pair.master.try_clone_reader()?;
        let sid = session_id.clone();
        
        std::thread::spawn(move || {
            let mut buf = [0u8; 4096];
            loop {
                match reader.read(&mut buf) {
                    Ok(n) if n > 0 => {
                        let payload = PtyDataPayload {
                            id: sid.clone(),
                            data: buf[0..n].to_vec(),
                        };
                        // Emit to frontend (reuse "ssh-data" event or create new "term-data"?)
                        // Reusing "ssh-data" simplifies frontend logic as it listens for generic binary data
                        if let Err(_) = window.emit("ssh-data", payload) {
                            break; // Window closed
                        }
                    }
                    Ok(_) => break, // EOF
                    Err(_) => break, // Error
                }
            }
            // Optional: Emit exit status
        });

        let writer = pair.master.take_writer()?;

        Ok(Self {
            master: Arc::new(Mutex::new(pair.master)),
            writer: Arc::new(Mutex::new(writer)),
        })
    }

    pub fn resize(&self, rows: u16, cols: u16) -> Result<()> {
        let master = self.master.lock().unwrap();
        master.resize(PtySize {
            rows,
            cols,
            pixel_width: 0,
            pixel_height: 0,
        }).map_err(|e| anyhow!("Resize failed: {}", e))
    }

    pub fn write(&self, data: &[u8]) -> Result<()> {
        let mut writer = self.writer.lock().unwrap();
        writer.write_all(data).map_err(|e| anyhow!("Write failed: {}", e))
    }
}