// FILE: ui/js/app.js
// ------------------

const invoke = window.__TAURI__.core.invoke;
let serialId = "S_" + Math.random().toString(36).substr(2, 5);

const App = {
    serialTerm: null,
    sshTerm: null,
    serialFit: null,
    sshFit: null,
    serialId: null,
};

document.addEventListener('DOMContentLoaded', async () => {
    console.log("App Starting...");
    
    // 1. Inject Templates
    if (window.Templates && typeof window.Templates.inject === 'function') {
        window.Templates.inject();
        if (typeof window.Templates.initSecurityUI === 'function') {
            window.Templates.initSecurityUI();
        }
    }

    // 2. Initialize Wizards
    if (typeof ConnWizard !== 'undefined') ConnWizard.init();
    
    // 3. Security Check
    await checkLockStatus();

    // 4. Global Events
    window.addEventListener('dragover', (e) => e.preventDefault());
    window.addEventListener('drop', (e) => e.preventDefault());

    // Prevent default refresh behavior (F5 or Ctrl+R or Cmd+R)
    document.addEventListener('keydown', (e) => {
        if (e.key === 'F5' || ((e.ctrlKey || e.metaKey) && e.key.toLowerCase() === 'r')) {
            e.preventDefault();
            
            // Allow Ctrl+R to act as "Refresh Folder" ONLY if we are in SFTP view
            const sftpView = document.getElementById('view-sftp');
            if (sftpView && sftpView.classList.contains('active')) {
                console.log("[App] Intercepted Ctrl+R -> Refreshing SFTP");
                if (typeof window.refreshSftp === 'function') {
                    window.refreshSftp(undefined, false);
                }
            } else {
                console.log("[App] Intercepted Refresh -> Ignored to preserve state");
            }
        }
    });

    // 5. Initialize UI
    if(typeof initTabs === 'function') initTabs();
    if(typeof initSerial === 'function') initSerial();
    if(typeof initSSH === 'function') initSSH(); 
    if(typeof initSFTP === 'function') initSFTP(); 
    
    if (window.Connections && typeof window.Connections.init === 'function') {
        window.Connections.init();
    }
    
    if(typeof startPoller === 'function') startPoller(); 
    
    // 6. Bindings
    Utils.bindEnterKey('unlock-pass', unlockApp);
    const unlockBtn = document.getElementById('btn-unlock');
    if(unlockBtn) unlockBtn.onclick = unlockApp;

    const serialToggle = document.getElementById('serial-settings-toggle');
    if(serialToggle) serialToggle.onclick = () => Utils.toggle('serial-settings-drawer');

    const aboutBtn = document.getElementById('btn-about');
    if (aboutBtn) {
        aboutBtn.onclick = (e) => {
            e.preventDefault(); 
            const modal = document.getElementById('about-modal');
            if(modal) {
                modal.style.display = 'flex';
                if (window.Templates && typeof window.Templates.refreshSecurityStatus === 'function') {
                    window.Templates.refreshSecurityStatus();
                }
            }
        };
    }

    // 7. Listeners
    await listen('serial-data', e => {
        const p = e.payload;
        if(p.id === AppState.serial.activeSessionId && AppState.serial.term) {
            AppState.serial.term.write(new Uint8Array(p.data));
        }
    });

    await listen('serial-status', e => {
        const p = e.payload;
        if(p.id === AppState.serial.activeSessionId) {
            if (typeof Serial !== 'undefined' && Serial.updateStatus) Serial.updateStatus(p.status);
        }
    });

    await listen('app-close-request', async () => {
        if (typeof SftpEditor !== 'undefined' && SftpEditor.isDirty) {
            const exit = await Modal.confirm(
                "You have unsaved changes in the SFTP editor.\n\nExit anyway?",
                "Unsaved Changes"
            );
            if (!exit) return; 
        }
        invoke('force_app_exit');
    });

    // 8. Prevent Navigation away if dirty
    document.body.addEventListener('click', async (e) => {
        if (typeof SftpEditor === 'undefined' || !SftpEditor.isDirty) return;

        const tab = e.target.closest('.tab');
        if (tab && !tab.classList.contains('active') && tab.dataset.target !== 'view-sftp') {
            e.preventDefault();
            e.stopPropagation();
            const discard = await Modal.confirm("You have unsaved changes. Discard them?", "Unsaved Changes");
            if (discard) {
                SftpEditor.setDirty(false);
                tab.click();
            }
        }
    }, true);

    console.log("Initialization Complete.");
});

async function checkLockStatus() {
    try {
        const isLocked = await invoke('get_lock_status');
        const overlay = document.getElementById('lock-screen');
        if (isLocked) {
            overlay.style.display = 'flex';
            const input = document.getElementById('unlock-pass');
            if(input) input.focus();
        } else {
            overlay.style.display = 'none';
            await performLoad();
        }
    } catch(e) {
        console.error("Lock check failed:", e);
    }
}

async function unlockApp() {
    const passField = document.getElementById('unlock-pass');
    const msg = document.getElementById('unlock-msg');
    const pass = passField.value;

    if(!pass) return;

    try {
        const success = await invoke('unlock_app', { password: pass });
        if(success) {
            document.getElementById('lock-screen').style.display = 'none';
            passField.value = "";
            msg.innerText = "";
            
            console.log("[Unlock] Success. Reloading store...");
            await performLoad();

        } else {
             msg.innerText = "Internal Error";
        }
    } catch(e) {
        msg.innerText = "Invalid Password";
        passField.select();
    }
}

// FIX: Explicitly refresh Serial List after store load
async function performLoad() {
    if (window.ConnStore) {
        await window.ConnStore.load();
        
        // Ensure Serial names are updated immediately
        if (typeof Serial !== 'undefined') {
            console.log("[App] Refreshing serial list with loaded names...");
            Serial.refreshList();
        }

        if (typeof ConnRender !== 'undefined') {
            ConnRender.render();
        } else if (window.Connections && window.Connections.render) {
            window.Connections.render();
        }
    }
}

function logToTerminal(msg) {
    const formatted = `\r\n\x1b[2m[LOG]: ${msg}\x1b[0m`;
    if(AppState.serial.term) AppState.serial.term.writeln(formatted);
}