// FILE: ui/js/conn/render.js
// --------------------------

window.ConnRender = {
    dragSrcId: null,

    render: () => {
        const savedList = document.getElementById('shelf-saved-list');
        const activeList = document.getElementById('shelf-active-list');
        
        if (!savedList || !activeList) return;

        savedList.innerHTML = '';
        activeList.innerHTML = '';

        const store = window.ConnStore;
        if (!store) return;
        
        // FIX: Exclude 'serial' type from Saved Shelf (Metadata only)
        const savedItems = store.items.filter(i => 
            i.status === 'Offline' && 
            i.type !== 'local' && 
            i.type !== 'serial'
        );
        
        const activeItems = store.items.filter(i => i.status !== 'Offline' || i.type === 'local');

        const sshView = document.getElementById('view-ssh');
        const sftpView = document.getElementById('view-sftp');
        let activeSshId = null;
        let activeSftpHostId = null;

        if (sshView && sshView.classList.contains('active')) {
            activeSshId = window.AppState.ssh.activeSessionId;
        } else if (sftpView && sftpView.classList.contains('active')) {
            if (typeof SftpTabManager !== 'undefined') {
                const tab = SftpTabManager.getActive();
                if (tab) activeSftpHostId = tab.hostId;
            }
        }

        const filterText = (window.Connections) ? window.Connections.filterText : "";
        const filteredSaved = savedItems.filter(i => 
            i.name.toLowerCase().includes(filterText) ||
            (i.hostData && i.hostData.hostname.includes(filterText))
        );

        if (filteredSaved.length === 0) {
            savedList.innerHTML = `<li class="empty-state">No profiles</li>`;
        } else {
            filteredSaved.forEach(item => {
                const li = document.createElement('li');
                const textColor = (item.hostData && item.hostData.color) ? item.hostData.color : '#e1e1e1';
                
                li.innerHTML = `
                    <div class="saved-item-row" style="display:flex; justify-content:space-between; width:100%">
                        <span class="saved-name" style="color:${textColor}; font-weight:bold;">${item.name}</span>
                        <div class="saved-actions" style="display:none; gap:5px;">
                            <div class="tiny-btn btn-edit">✎</div>
                            <div class="tiny-btn btn-del">✖</div>
                        </div>
                    </div>
                    <div class="saved-meta" style="font-size:10px; color:#666;">
                        ${item.hostData ? item.hostData.hostname : 'Unknown'}
                    </div>
                `;
                
                li.onmouseenter = () => li.querySelector('.saved-actions').style.display = 'flex';
                li.onmouseleave = () => li.querySelector('.saved-actions').style.display = 'none';
                
                li.onclick = () => { if (window.Connections) window.Connections.connectSaved(item); };
                li.querySelector('.btn-edit').onclick = (e) => { 
                    e.stopPropagation(); 
                    if (window.Connections) window.Connections.editSaved(item); 
                };
                li.querySelector('.btn-del').onclick = (e) => { 
                    e.stopPropagation(); 
                    if (window.Connections) window.Connections.deleteSaved(item); 
                };
                savedList.appendChild(li);
            });
        }

        if (activeItems.length === 0) {
            activeList.innerHTML = `<div style="padding:20px; color:#666; font-style:italic;">No active sessions</div>`;
        } else {
            activeItems.forEach(item => {
                const card = document.createElement('div');
                card.className = 'connection-card';
                card.dataset.id = item.id;
                card.setAttribute('draggable', 'true');
                
                card.addEventListener('dragstart', (e) => {
                    window.ConnRender.dragSrcId = item.id;
                    e.dataTransfer.effectAllowed = 'move';
                    e.dataTransfer.setData('text/plain', item.id);
                    setTimeout(() => card.style.opacity = '0.4', 0);
                });

                card.addEventListener('dragend', () => {
                    card.style.opacity = '1';
                    document.querySelectorAll('.connection-card').forEach(c => c.style.border = '1px solid #3e3e42');
                });

                card.addEventListener('dragover', (e) => {
                    if (e.preventDefault) e.preventDefault();
                    e.dataTransfer.dropEffect = 'move';
                    card.style.borderColor = '#007acc';
                    return false;
                });

                card.addEventListener('dragleave', () => {
                    card.style.borderColor = '#3e3e42';
                });

                card.addEventListener('drop', (e) => {
                    e.stopPropagation();
                    const srcId = window.ConnRender.dragSrcId;
                    const targetId = item.id;
                    if (srcId && srcId !== targetId && window.ConnStore) {
                        window.ConnStore.reorderActive(srcId, targetId);
                        window.ConnRender.render();
                    }
                    return false;
                });

                let isActive = false;
                if (activeSshId && item.id === activeSshId) isActive = true;
                if (activeSftpHostId && (item.hostId === activeSftpHostId || item.id === activeSftpHostId)) isActive = true;
                if (isActive) card.classList.add('active-in-view');

                let statusColor = '#50fa7b'; 
                if (item.status === 'Error') statusColor = '#ff5555'; 
                else if (item.status === 'Connecting...') statusColor = '#ffb86c'; 
                else if (item.type === 'local') statusColor = '#8be9fd'; 

                const endVisibility = item.type === 'local' ? 'visibility:hidden' : '';
                let cardColor = item.hostData && item.hostData.color ? item.hostData.color : '#fff';
                
                if (!item.hostData && item.hostId && window.ConnStore) {
                    const saved = window.ConnStore.items.find(i => i.savedId === item.hostId);
                    if (saved && saved.hostData) cardColor = saved.hostData.color;
                }

                card.innerHTML = `
                    <div class="conn-header">
                        <span title="${item.name}" style="white-space:nowrap; overflow:hidden; text-overflow:ellipsis; max-width: 120px; color:${cardColor};">${item.name}</span>
                    </div>
                    <div class="conn-status" style="color:${statusColor}">${item.status}</div>
                    <div class="conn-actions">
                        <div class="conn-btn btn-term">Term</div>
                        <div class="conn-btn btn-sftp">SFTP</div>
                        <div class="conn-btn btn-end" style="color:#ff5555; ${endVisibility}">End</div>
                    </div>
                `;

                card.querySelector('.btn-term').onclick = (e) => {
                    if (window.Connections) window.Connections.focusTerminal(item.id, e.altKey);
                };
                card.querySelector('.btn-sftp').onclick = (e) => {
                    if (window.Connections) window.Connections.focusSftp(item.id, e.altKey);
                };
                card.querySelector('.btn-end').onclick = () => {
                    if (window.Connections) window.Connections.disconnect(item.id);
                };

                activeList.appendChild(card);
            });
        }
    }
};

const ConnRender = window.ConnRender;