// FILE: ui/js/conn/store.js
// -------------------------

window.ConnStore = {
    items: [],

    load: async () => {
        console.log("[ConnStore] Loading...");
        const activeSessions = window.ConnStore.items.filter(i => i.status !== 'Offline');
        
        try {
            const config = await invoke('get_app_config');
            window.ConnStore.items = [];

            // 1. Add Local Static
            let localItem = activeSessions.find(i => i.id === "local_static");
            if (!localItem) {
                localItem = {
                    id: "local_static",
                    name: "This Computer",
                    type: "local",
                    status: "Ready",
                    hostId: "LOCAL", 
                    hostData: { hostname: "localhost", username: "user", color: "#007acc" }
                };
            }
            window.ConnStore.items.push(localItem);

            // 2. Add other Active Sessions
            activeSessions.forEach(sess => {
                if (sess.id !== "local_static") window.ConnStore.items.push(sess);
            });

            // 3. Add Saved SSH Hosts
            if (config && typeof config.hosts === 'object') {
                const hosts = Object.values(config.hosts);
                hosts.sort((a, b) => a.name.localeCompare(b.name, undefined, { sensitivity: 'base' }));
                hosts.forEach(host => {
                    window.ConnStore.items.push({
                        id: host.id,
                        name: host.name,
                        type: 'ssh',
                        status: 'Offline',
                        savedId: host.id,
                        hostData: host
                    });
                });
            }

            // 4. Add Saved Serial Devices
            if (config && typeof config.serial_instances === 'object') {
                const serials = Object.entries(config.serial_instances);
                serials.forEach(([instanceId, info]) => {
                    window.ConnStore.items.push({
                        id: instanceId,
                        name: info.friendly_name || "Serial Device",
                        type: 'serial',
                        status: 'Offline',
                        savedId: instanceId,
                        hostData: {
                            hostname: "COM/TTY",
                            color: info.color || "#e5e510",
                            profileId: info.profile_id
                        }
                    });
                });
            }

        } catch (e) {
            console.warn("[ConnStore] Load failed (Locked?):", e);
            if (window.ConnStore.items.length === 0) {
                window.ConnStore.items.push({
                    id: "local_static",
                    name: "This Computer",
                    type: "local",
                    status: "Ready",
                    hostId: "LOCAL"
                });
            }
        }
    },

    // BATCH 6: New method for Serial Persistence
    saveSerial: async (instanceId, name, color) => {
        try {
            await invoke('save_serial_device', { 
                instanceId, 
                friendlyName: name, 
                color 
            });
            // Reload to refresh UI lists
            await window.ConnStore.load();
            // Refresh Serial View if active
            if (typeof Serial !== 'undefined' && typeof Serial.refreshList === 'function') {
                Serial.refreshList();
            }
        } catch(e) {
            console.error("Failed to save serial info", e);
            alert("Error saving serial device: " + e);
        }
    },

    addActive: (sessionObj) => {
        const store = window.ConnStore;
        if (store.items.find(i => i.id === sessionObj.id)) return;
        const firstOfflineIdx = store.items.findIndex(i => i.status === 'Offline');
        const newItem = { ...sessionObj, status: 'Connected', timestamp: Date.now() };
        if (firstOfflineIdx === -1) store.items.push(newItem);
        else store.items.splice(firstOfflineIdx, 0, newItem);
    },

    reorderActive: (srcId, targetId) => {
        const store = window.ConnStore;
        const srcIdx = store.items.findIndex(i => i.id === srcId);
        const tgtIdx = store.items.findIndex(i => i.id === targetId);
        if (srcIdx === -1 || tgtIdx === -1) return;
        const [movedItem] = store.items.splice(srcIdx, 1);
        store.items.splice(tgtIdx, 0, movedItem);
    },

    removeById: (id) => {
        if (id === "local_static") return;
        window.ConnStore.items = window.ConnStore.items.filter(i => i.id !== id);
    },

    updateStatus: (id, status) => {
        const item = window.ConnStore.items.find(i => i.id === id);
        if (item) item.status = status;
    },

    find: (id) => window.ConnStore.items.find(i => i.id === id),

    getSavedByHostUser: (host, user) => {
        return window.ConnStore.items.find(i => 
            i.status === 'Offline' && i.hostData && 
            i.hostData.hostname === host && i.hostData.username === user
        );
    },
    
    getSerialInfo: (instanceId) => {
        return window.ConnStore.items.find(i => 
            i.type === 'serial' && i.savedId === instanceId
        );
    }
};

const ConnStore = window.ConnStore;