// FILE: ui/js/modal.js
// --------------------

const Modal = {
    _format: (text) => {
        if (!text) return "";
        let safe = text.replace(/&/g, "&amp;").replace(/</g, "&lt;").replace(/>/g, "&gt;")
                       .replace(/"/g, "&quot;").replace(/'/g, "&#039;");
        safe = safe.replace(/\*\*(.*?)\*\*/g, '<strong>$1</strong>');
        safe = safe.replace(/\n/g, '<br>');
        return safe;
    },

    confirm: (message, title="Confirmation") => {
        return new Promise((resolve) => {
            const overlay = document.getElementById('modal-overlay');
            const titleEl = document.getElementById('modal-title');
            const bodyEl = document.getElementById('modal-body');
            const btnOk = document.getElementById('modal-btn-ok');
            const btnCancel = document.getElementById('modal-btn-cancel');

            if (!overlay) return resolve(false);

            titleEl.innerText = title;
            bodyEl.innerHTML = Modal._format(message);
            
            // Ensure Cancel is visible for confirm
            btnCancel.style.display = 'inline-block';
            btnOk.innerText = "OK";
            overlay.style.display = 'flex';

            const close = (result) => {
                overlay.style.display = 'none';
                resolve(result);
            };

            // Deep clone to wipe previous listeners, then re-attach
            const newOk = btnOk.cloneNode(true);
            const newCancel = btnCancel.cloneNode(true);
            
            newOk.onclick = (e) => { e.preventDefault(); e.stopPropagation(); close(true); };
            newCancel.onclick = (e) => { e.preventDefault(); e.stopPropagation(); close(false); };
            
            btnOk.parentNode.replaceChild(newOk, btnOk);
            btnCancel.parentNode.replaceChild(newCancel, btnCancel);
            
            newOk.focus();

            const outsideClick = (e) => {
                if (e.target === overlay) {
                    close(false); 
                    overlay.removeEventListener('click', outsideClick);
                }
            };
            overlay.addEventListener('click', outsideClick);
        });
    },

    alert: (message, title="Alert") => {
        return new Promise((resolve) => {
            const overlay = document.getElementById('modal-overlay');
            const titleEl = document.getElementById('modal-title');
            const bodyEl = document.getElementById('modal-body');
            const btnOk = document.getElementById('modal-btn-ok');
            const btnCancel = document.getElementById('modal-btn-cancel');

            if (!overlay) return resolve();

            titleEl.innerText = title;
            bodyEl.innerHTML = Modal._format(message);
            
            // Hide cancel for alert
            btnCancel.style.display = 'none'; 
            btnOk.innerText = "Close";
            overlay.style.display = 'flex';

            // Clone to strip old listeners
            const newOk = btnOk.cloneNode(true);
            
            const close = (e) => {
                if(e) { e.preventDefault(); e.stopPropagation(); }
                overlay.style.display = 'none';
                resolve();
            };
            
            newOk.onclick = close;
            btnOk.parentNode.replaceChild(newOk, btnOk);
            newOk.focus();

            const outsideClick = (e) => {
                if (e.target === overlay) {
                    close(); 
                    overlay.removeEventListener('click', outsideClick);
                }
            };
            overlay.addEventListener('click', outsideClick);
        });
    },

    prompt: (message, defaultValue="") => {
        // Native prompt is blocking, which is fine for simple inputs, 
        // but we might want a custom one later. keeping native for now.
        return Promise.resolve(window.prompt(message, defaultValue));
    }
};

// Global listener for the Connection Wizard Modal
document.addEventListener('DOMContentLoaded', () => {
    // Helper to close specific modals by clicking outside
    const bindOutsideClose = (id, closeFn) => {
        const el = document.getElementById(id);
        if(el) {
            el.addEventListener('click', (e) => {
                if(e.target === el) closeFn();
            });
        }
    };

    // Re-bind Wizard Close (Safe check)
    if (typeof ConnWizard !== 'undefined') {
        bindOutsideClose('connection-wizard-modal', ConnWizard.hide);
    }

    // Re-bind Sync Wizard Close
    if (typeof SyncWizard !== 'undefined') {
        bindOutsideClose('sync-wizard-modal', SyncWizard.hide);
    }
});