// FILE: ui/js/sftp/render.js
// --------------------------

const SftpRender = {
    isInitialized: false,
    currentPathRef: ".", 

    list: (targetElement, files, currentPath, callbacks) => {
        if (!targetElement) return;
        
        SftpRender.currentPathRef = currentPath; 

        if (!targetElement._dragInitialized) {
            if (typeof SftpDrag !== 'undefined') {
                SftpDrag.init(targetElement, callbacks);
            }
            targetElement._dragInitialized = true;
        }
        
        targetElement._callbacks = callbacks;
        targetElement._currentPath = currentPath;

        targetElement.innerHTML = '';

        if (currentPath !== "/") {
            const li = document.createElement('li');
            const parentPath = SftpUtils.getParentPath(currentPath) || "/";
            
            // BATCH 3 FIX: Add Selection Handle for ".." row too (consistent spacing)
            li.innerHTML = `
                <div class="sftp-handle" style="visibility:hidden"></div>
                <span class="sftp-name text-accent">📁 ..</span>
            `;
            
            li.dataset.path = parentPath; 
            li.dataset.isdir = "true";
            
            li.onclick = () => {
                callbacks.onNavigate(parentPath);
            };
            
            li.addEventListener('dragover', (e) => {
                e.preventDefault(); 
                e.dataTransfer.dropEffect = "copy";
                li.classList.add('drag-over');
            });
            li.addEventListener('dragleave', () => li.classList.remove('drag-over'));
            
            targetElement.appendChild(li);
        }

        if (!files || files.length === 0) {
            const li = document.createElement('li');
            li.className = "empty-state";
            li.innerText = "(Empty - Drop files here)";
            targetElement.appendChild(li);
            return;
        }

        files.forEach((f, index) => {
            const li = document.createElement('li');
            
            const icon = f.is_dir ? "📁" : "📄";
            const colorClass = f.is_dir ? "text-accent" : "text-white";
            const sizeStr = f.is_dir ? "" : SftpUtils.formatShortSize(f.size);
            
            let fullPath = f.name;
            if (currentPath === "/") fullPath = "/" + f.name;
            else if (currentPath === ".") fullPath = f.name;
            else if (currentPath.endsWith("/")) fullPath = currentPath + f.name;
            else fullPath = currentPath + "/" + f.name;

            li.dataset.index = index;
            li.dataset.path = fullPath;
            li.dataset.isdir = f.is_dir;
            
            if (f.name.startsWith('.')) li.classList.add('dotfile');
            
            li.setAttribute("draggable", "true");
            li.title = f.name;
            
            li.addEventListener('dragstart', (e) => {
                e.dataTransfer.effectAllowed = "copyMove";
                e.dataTransfer.setData("text/plain", "internal"); 
                if (callbacks.onDragStart) {
                    callbacks.onDragStart(fullPath, f.name);
                }
                setTimeout(() => li.style.opacity = "0.5", 0);
            });
            
            li.addEventListener('dragend', (e) => {
                li.style.opacity = "1";
            });

            li.addEventListener('click', async (e) => {
                if (e.target.classList.contains('del-btn') || e.target.tagName === 'INPUT') return;
                
                // BATCH 3 FIX: Handle Click Interception
                if (e.target.classList.contains('sftp-handle')) {
                    e.stopPropagation();
                    e.preventDefault();
                    // Toggle selection logic for handle
                    if (SftpUI.selection.has(fullPath)) {
                        SftpUI.removeFromSelection(fullPath);
                    } else {
                        SftpUI.addToSelection(fullPath);
                        SftpUI.lastFocusedPath = fullPath;
                    }
                    return;
                }

                targetElement.focus();

                if (e.altKey) {
                    e.preventDefault(); e.stopPropagation();
                    const originalBg = li.style.backgroundColor;
                    li.style.backgroundColor = "#007acc";
                    const hostId = SftpUtils.getActiveSshId();
                    try {
                        await invoke('sftp_to_clipboard', { hostId, path: fullPath });
                        li.style.backgroundColor = "#28a745";
                        setTimeout(() => { li.style.backgroundColor = originalBg; }, 500);
                    } catch(err) {
                        li.style.backgroundColor = "#dc3545";
                        setTimeout(() => { li.style.backgroundColor = originalBg; }, 500);
                    }
                    return;
                }

                if (e.ctrlKey || e.metaKey) {
                    if (SftpUI.selection.has(fullPath)) {
                        SftpUI.removeFromSelection(fullPath);
                    } else {
                        SftpUI.addToSelection(fullPath);
                        SftpUI.lastFocusedPath = fullPath;
                    }
                } 
                else if (e.shiftKey && SftpUI.lastFocusedPath) {
                    const allItems = Array.from(targetElement.querySelectorAll('li:not(.empty-state)'));
                    const fileItems = allItems.filter(i => i.dataset.path); 
                    
                    let startIdx = fileItems.findIndex(i => i.dataset.path === SftpUI.lastFocusedPath);
                    let endIdx = fileItems.findIndex(i => i.dataset.path === fullPath);
                    
                    if (startIdx !== -1 && endIdx !== -1) {
                        const low = Math.min(startIdx, endIdx);
                        const high = Math.max(startIdx, endIdx);
                        for (let i = low; i <= high; i++) {
                            SftpUI.selection.add(fileItems[i].dataset.path);
                        }
                        SftpUI.redrawSelection();
                    }
                } 
                else {
                    // Standard Click: Select Only This, and Execute Action
                    SftpUI.setSelection(fullPath);
                    SftpUI.lastFocusedPath = fullPath;

                    if (f.is_dir) {
                        callbacks.onNavigate(fullPath);
                    } else {
                        callbacks.onOpen(fullPath, f.size);
                    }
                }
            });

            li.addEventListener('contextmenu', (e) => {
                e.preventDefault();
                targetElement.focus();
                
                if (!SftpUI.selection.has(fullPath)) {
                    SftpUI.setSelection(fullPath);
                    SftpUI.lastFocusedPath = fullPath;
                }
                
                callbacks.onContextMenu(e.clientX, e.clientY, {
                    name: f.name, path: fullPath, isDir: f.is_dir
                });
            });

            // BATCH 3 FIX: Insert Handle
            li.innerHTML = `
                <div style="flex-grow:1; display:flex; overflow:hidden; align-items:center;">
                    <div class="sftp-handle" title="Select row"></div>
                    <span class="sftp-name ${colorClass}" title="${f.name}">${icon} ${f.name}</span>
                </div>
                <div style="display:flex; gap:10px; align-items:center;">
                    <span class="sftp-size">${sizeStr}</span>
                    <button class="del-btn" title="Delete">✖</button>
                </div>
            `;

            const delBtn = li.querySelector('.del-btn');
            if (delBtn) {
                delBtn.onclick = (e) => {
                    e.stopPropagation();
                    callbacks.onDelete(fullPath, f.is_dir, false);
                };
            }

            targetElement.appendChild(li);
        });
    },
};