// FILE: ui/js/serial.js
// ---------------------

const Serial = {
    debounceTimer: null,
    isConnecting: false,
    
    // Maps instanceId -> sessionId
    sessions: {}, 

    init: () => {
        if (typeof Terminal === 'undefined') return;

        AppState.serial.term = new Terminal({
            theme: { background: '#000000' },
            fontFamily: 'Consolas, monospace',
            fontSize: 14,
            cursorBlink: true,
            wordWrap: true,
            allowProposedApi: true
        });

        if (typeof FitAddon !== 'undefined') {
            AppState.serial.fit = new FitAddon.FitAddon();
            AppState.serial.term.loadAddon(AppState.serial.fit);
        }

        const container = document.getElementById('serial-terminal');
        if(container) {
            AppState.serial.term.open(container);
            if(AppState.serial.fit) AppState.serial.fit.fit();
        }

        AppState.serial.term.writeln('\x1b[36m--- OmniTerm Serial Ready ---\x1b[0m');
        
        window.addEventListener('resize', () => {
            clearTimeout(Serial.debounceTimer);
            Serial.debounceTimer = setTimeout(() => {
                if(AppState.serial.fit) AppState.serial.fit.fit();
            }, 200);
        });

        const sendFn = (data) => {
            if (AppState.serial.activeSessionId) {
                const prefixRaw = document.getElementById('serial-tx-prefix').value;
                const suffixRaw = document.getElementById('serial-tx-suffix').value;
                const localEcho = document.getElementById('serial-local-echo').checked;

                const processEscapes = (str) => {
                    return str.replace(/\\r/g, '\r')
                              .replace(/\\n/g, '\n')
                              .replace(/\\t/g, '\t')
                              .replace(/\\x([0-9A-Fa-f]{2})/g, (_, hex) => String.fromCharCode(parseInt(hex, 16)));
                };

                const prefix = processEscapes(prefixRaw);
                const suffix = processEscapes(suffixRaw);
                const payload = prefix + data + suffix;

                if (localEcho) {
                    AppState.serial.term.write(payload);
                }

                invoke('send_serial_data', { 
                    sessionId: AppState.serial.activeSessionId, 
                    data: payload 
                });
            }
        };

        AppState.serial.term.onData(sendFn);
        Utils.enableMobaStyleControls(AppState.serial.term, sendFn);

        const refreshBtn = document.getElementById('refresh-btn');
        if (refreshBtn) refreshBtn.onclick = Serial.refreshList;

        document.getElementById('chk-dtr').onchange = (e) => Serial.togglePin("dtr", e.target.checked);
        document.getElementById('chk-rts').onchange = (e) => Serial.togglePin("rts", e.target.checked);
        
        document.getElementById('setting-baud').onchange = (e) => {
            document.getElementById('serial-baud-display').innerText = e.target.value;
        };

        Serial.refreshList();
    },

    refreshList: async () => {
        const list = document.getElementById('serial-list');
        list.innerHTML = '<li class="empty-state">Scanning...</li>';
        try {
            const devs = await invoke('list_serial_devices');
            list.innerHTML = '';
            if (!devs || devs.length === 0) {
                list.innerHTML = '<li class="empty-state">No devices found</li>';
                return;
            }
            
            const savedItems = (window.ConnStore && window.ConnStore.items) ? window.ConnStore.items : [];

            devs.forEach(d => {
                const li = document.createElement('li');
                
                const saved = savedItems.find(i => i.type === 'serial' && i.savedId === d.instance_id);
                
                // Use saved name if available, else system friendly name
                let friendly = saved ? saved.name : d.friendly_name;
                const redundantRegex = new RegExp(`\\(${d.port_name}\\)$`);
                friendly = friendly.replace(redundantRegex, "").trim();

                const isConnected = Serial.sessions[d.instance_id] !== undefined;
                const isActive = isConnected && AppState.serial.activeSessionId === Serial.sessions[d.instance_id];
                
                let borderStyle = "";
                if (isActive) borderStyle = "border-left: 3px solid #007acc; background-color: #37373d;";
                else if (isConnected) borderStyle = "border-left: 3px solid #555; opacity: 0.8;";

                li.innerHTML = `
                    <div class="dev-row" style="flex-grow:1; ${borderStyle}">
                        <span class="dev-name">${d.port_name}</span>
                        <span class="dev-detail" title="${friendly}">${friendly}</span>
                    </div>
                    <button class="icon-btn edit-btn" style="font-size:12px; margin-left:5px; opacity:0.5;" title="Rename">✎</button>
                `;
                
                li.onclick = () => Serial.connect(d);
                
                // Rename Handler
                const editBtn = li.querySelector('.edit-btn');
                editBtn.onclick = (e) => {
                    e.stopPropagation();
                    const newName = prompt("Set Friendly Name for " + d.port_name, friendly);
                    if (newName !== null) {
                        if (window.ConnStore) {
                            window.ConnStore.saveSerial(d.instance_id, newName || d.friendly_name, null);
                        }
                    }
                };
                
                li.onmouseenter = () => { editBtn.style.opacity = "1"; };
                li.onmouseleave = () => { editBtn.style.opacity = "0.5"; };

                list.appendChild(li);
            });
        } catch (e) {
            list.innerHTML = `<li class="empty-state" style="color:red">Error: ${e}</li>`;
        }
    },

    connect: async (device) => {
        // Multi-session logic
        if (Serial.sessions[device.instance_id]) {
            const existingSessionId = Serial.sessions[device.instance_id];
            
            console.log(`[Serial] Switching to active session: ${existingSessionId}`);
            AppState.serial.activeSessionId = existingSessionId;
            AppState.serial.term.reset(); 
            AppState.serial.term.writeln(`\x1b[36m--- Switched to ${device.port_name} ---\x1b[0m`);
            
            Serial.refreshList();
            
            const el = document.getElementById('serial-status-bar');
            if(el) el.innerText = "Connected";
            const termContainer = document.getElementById('serial-terminal');
            if(termContainer) termContainer.classList.remove('serial-disconnected');
            
            return;
        }

        if (Serial.isConnecting) {
            console.log("Connection guard active...");
            return;
        }

        Serial.isConnecting = true;
        
        AppState.serial.term.reset();
        AppState.serial.term.writeln(`Connecting to ${device.port_name}...`);
        
        const termContainer = document.getElementById('serial-terminal');
        if(termContainer) termContainer.classList.remove('serial-disconnected');

        const baudVal = parseInt(document.getElementById('setting-baud').value) || 115200;
        
        const newSessionId = "sess_" + Math.random().toString(36).substr(2, 9);
        AppState.serial.activeSessionId = newSessionId;
        
        try {
            const el = document.getElementById('serial-status-bar');
            if(el) el.innerText = "Connecting...";
            
            await invoke('start_serial_session', {
                profileId: "quick_connect",
                deviceInstanceId: device.instance_id,
                sessionId: newSessionId,
                baudRate: baudVal
            });
            
            // Register session mapping
            Serial.sessions[device.instance_id] = newSessionId;
            Serial.refreshList();

            const prefix = document.getElementById('serial-tx-prefix').value || "(none)";
            const suffix = document.getElementById('serial-tx-suffix').value.replace(/\r/g,'\\r').replace(/\n/g,'\\n');
            
            AppState.serial.term.writeln(`\x1b[2m[Config] Baud: ${baudVal} | Pfx: ${prefix} | Sfx: ${suffix}\x1b[0m`);
        } catch (e) {
            AppState.serial.term.writeln(`\r\nError: ${e}`);
            // Cleanup on fail
            delete Serial.sessions[device.instance_id];
        } finally {
            Serial.isConnecting = false;
        }
    },

    togglePin: (pin, enabled) => {
        if(AppState.serial.activeSessionId) {
            invoke('toggle_serial_pin', {
                sessionId: AppState.serial.activeSessionId, pin, enabled
            });
        }
    },

    updateStatus: (status) => {
        const el = document.getElementById('serial-status-bar');
        const termContainer = document.getElementById('serial-terminal');

        if (el && el.innerText !== status) {
            el.innerText = status;
        }

        // Cleanup map if disconnected
        if (status.includes("Error") || status === "Disconnected") {
            const activeId = AppState.serial.activeSessionId;
            if (activeId) {
                const entry = Object.entries(Serial.sessions).find(([_, sessId]) => sessId === activeId);
                if (entry) {
                    const [instanceId, _] = entry;
                    delete Serial.sessions[instanceId];
                    Serial.refreshList();
                }
            }
        }

        if (termContainer) {
            if (status === "Searching" || status.includes("Error") || status === "Disconnected") {
                if (!termContainer.classList.contains('serial-disconnected')) {
                    termContainer.classList.add('serial-disconnected');
                    if (AppState.serial.term) AppState.serial.term.writeln(`\r\n\x1b[31m[STATUS: ${status}]\x1b[0m`);
                }
            } else if (status === "Connected") {
                if (termContainer.classList.contains('serial-disconnected')) {
                    termContainer.classList.remove('serial-disconnected');
                    if (AppState.serial.term) AppState.serial.term.writeln(`\r\n\x1b[32m[STATUS: ${status}]\x1b[0m`);
                }
            }
        }
    }
};

const initSerial = Serial.init;