// FILE: ui/js/sftp/drag.js
// ------------------------

const SftpDrag = {
    init: (listElement, callbacks) => {
        if (!listElement) return; // SAFETY CHECK

        let lastTarget = null;

        const getFolderTarget = (e) => {
            const li = e.target.closest('li');
            if (li && li.dataset.isdir === "true") return li;
            return null;
        };

        listElement.addEventListener('dragover', (e) => {
            e.preventDefault(); 
            e.dataTransfer.dropEffect = "copy"; 

            const specificFolder = getFolderTarget(e);
            if (specificFolder) {
                if (lastTarget && lastTarget !== specificFolder) lastTarget.classList.remove('drag-over');
                specificFolder.classList.add('drag-over');
                lastTarget = specificFolder;
                listElement.style.borderColor = "transparent";
            } else {
                if (lastTarget) lastTarget.classList.remove('drag-over');
                lastTarget = null;
                listElement.style.borderColor = "#007acc"; 
            }
        });

        listElement.addEventListener('dragleave', (e) => {
            if (e.target === listElement) {
                listElement.style.borderColor = "transparent";
            }
        });

        listElement.addEventListener('drop', async (e) => {
            e.preventDefault();
            e.stopPropagation();

            if (lastTarget) lastTarget.classList.remove('drag-over');
            listElement.style.borderColor = "transparent";
            lastTarget = null;

            const specificFolder = getFolderTarget(e);
            const destPath = specificFolder ? specificFolder.dataset.path : listElement._currentPath;
            
            const cbs = listElement._callbacks;
            if (!cbs) return;

            if (e.dataTransfer.items && e.dataTransfer.items.length > 0) {
                if (e.dataTransfer.items[0].kind === 'file') {
                    const promises = [];
                    for (let i = 0; i < e.dataTransfer.items.length; i++) {
                        const item = e.dataTransfer.items[i];
                        if (item.webkitGetAsEntry) {
                            const entry = item.webkitGetAsEntry();
                            if (entry) {
                                promises.push(SftpDrag.traverseFileTree(entry, ""));
                            }
                        }
                    }
                    
                    const results = await Promise.all(promises);
                    const flatList = results.flat();
                    
                    if (flatList.length > 0 && cbs.onDropExternal) {
                        cbs.onDropExternal(flatList, destPath);
                    }
                    return;
                }
            }

            if (cbs.onDropInternal) {
                cbs.onDropInternal(destPath);
            }
        });
    },

    traverseFileTree: (entry, path) => {
        return new Promise((resolve) => {
            if (entry.isFile) {
                entry.file(file => {
                    file.uploadRelPath = path; 
                    resolve([file]);
                });
            } else if (entry.isDirectory) {
                const dirReader = entry.createReader();
                const entries = [];
                const read = () => {
                    dirReader.readEntries(async (results) => {
                        if (results.length) {
                            entries.push(...results);
                            read();
                        } else {
                            const promises = entries.map(e => 
                                SftpDrag.traverseFileTree(e, path ? path + "/" + entry.name : entry.name)
                            );
                            const recursed = await Promise.all(promises);
                            resolve(recursed.flat());
                        }
                    });
                };
                read();
            }
        });
    }
};