// FILE: ui/js/sftp/editor.js
// --------------------------

const SftpEditor = {
    currentFile: null,
    isDirty: false,
    originalContent: "",
    isSpellcheck: false,
    
    lineHeight: 21, 
    paddingVert: 20, 

    init: () => {
        const textarea = document.getElementById('sftp-textarea');
        const gutter = document.getElementById('sftp-gutter');
        const ghost = document.getElementById('sftp-measure-ghost');
        const spellBtn = document.getElementById('sftp-spellcheck-btn');

        if (textarea && gutter && ghost) {
            let resizeTimeout;
            window.addEventListener('resize', () => {
                clearTimeout(resizeTimeout);
                resizeTimeout = setTimeout(SftpEditor.updateGutter, 50);
            });

            textarea.addEventListener('input', () => {
                SftpEditor.updateGutter();
                const nowDirty = textarea.value !== SftpEditor.originalContent;
                if (nowDirty !== SftpEditor.isDirty) {
                    SftpEditor.setDirty(nowDirty);
                }
            });

            textarea.addEventListener('scroll', () => {
                gutter.scrollTop = textarea.scrollTop;
            });

            textarea.addEventListener('keydown', (e) => {
                if ((e.ctrlKey || e.metaKey) && e.key === 's') {
                    e.preventDefault();
                    SftpEditor.save();
                }
                if ((e.ctrlKey || e.metaKey) && e.key === 'r') {
                    e.preventDefault();
                    if(SftpEditor.currentFile) SftpEditor.load(SftpEditor.currentFile, 0, true);
                }
                if (e.key === 'Tab') {
                    e.preventDefault();
                    document.execCommand('insertText', false, "\t");
                    SftpEditor.updateGutter();
                }
            });
        }
        
        const saveBtn = document.getElementById('sftp-save-btn');
        if(saveBtn) saveBtn.onclick = SftpEditor.save;

        if (spellBtn) {
            spellBtn.onclick = () => {
                SftpEditor.isSpellcheck = !SftpEditor.isSpellcheck;
                textarea.spellcheck = SftpEditor.isSpellcheck;
                spellBtn.classList.toggle('active', SftpEditor.isSpellcheck);
                const val = textarea.value;
                textarea.value = "";
                textarea.value = val; 
                SftpEditor.updateGutter();
            };
        }
    },

    updateGutter: () => {
        const textarea = document.getElementById('sftp-textarea');
        const gutter = document.getElementById('sftp-gutter');
        const ghost = document.getElementById('sftp-measure-ghost');
        
        if(!textarea || !gutter || !ghost) return;

        ghost.style.width = textarea.clientWidth + "px";

        const text = textarea.value;
        const physicalLines = text.split('\n'); 
        const lineCount = physicalLines.length;
        
        let gutterHtml = "";
        
        for(let i=0; i<lineCount; i++) {
            const rawLine = physicalLines[i];
            ghost.textContent = rawLine + '\u200b'; 
            const height = ghost.clientHeight;
            const contentHeight = Math.max(0, height - SftpEditor.paddingVert);
            const rows = Math.max(1, Math.round(contentHeight / SftpEditor.lineHeight));
            
            gutterHtml += `<div>${i+1}</div>`;
            for(let j=1; j<rows; j++) {
                gutterHtml += `<div>&nbsp;</div>`;
            }
        }
        
        gutter.innerHTML = gutterHtml;
    },

    setDirty: (dirty) => {
        SftpEditor.isDirty = dirty;
        const ind = document.getElementById('sftp-unsaved-indicator');
        const btn = document.getElementById('sftp-save-btn');
        if (ind) ind.style.display = dirty ? 'inline' : 'none';
        if (btn) btn.disabled = !dirty;
    },

    checkDirty: () => {
        if (SftpEditor.isDirty) {
            return confirm("You have unsaved changes. Discard them?");
        }
        return true;
    },

    load: async (fullPath, sizeBytes, force = false) => {
        if (!force && !SftpEditor.checkDirty()) return;

        if (sizeBytes && sizeBytes > 100 * 1024 * 1024) {
            return alert("File is extremely large (>100MB). Even preview might be slow.");
        }

        SftpEditor.currentFile = fullPath;
        SftpEditor.setDirty(false);
        
        const wrapper = document.getElementById('sftp-editor-wrapper');
        const textarea = document.getElementById('sftp-textarea');
        const placeholder = document.getElementById('sftp-placeholder');
        const filenameLabel = document.getElementById('sftp-filename');
        const statusMsg = document.getElementById('sftp-status-msg');

        placeholder.style.display = 'none';
        wrapper.style.display = 'flex'; 
        
        textarea.value = "Loading...";
        textarea.disabled = true;
        filenameLabel.innerText = fullPath;
        statusMsg.innerText = "Downloading...";
        statusMsg.style.color = "#888";
        
        document.getElementById('sftp-gutter').innerHTML = "";

        const hostId = SftpUtils.getActiveSshId();

        try {
            const res = await invoke('sftp_read_file', { hostId: hostId, path: fullPath });
            
            const binString = atob(res.data);
            const bytes = new Uint8Array(binString.length);
            for (let i = 0; i < binString.length; i++) {
                bytes[i] = binString.charCodeAt(i);
            }
            let content = new TextDecoder('utf-8').decode(bytes);

            if (res.truncated) {
                content += "\n\n[!!! WARNING: FILE TRUNCATED - TOO LARGE FOR EDITOR !!!]";
                statusMsg.innerText = "Read-Only (Truncated)";
                statusMsg.style.color = "#f1fa8c";
                textarea.readOnly = true; 
            } else {
                textarea.readOnly = false;
                statusMsg.innerText = "Ready";
            }

            SftpEditor.originalContent = content;
            textarea.value = content;
            textarea.disabled = false;
            
            setTimeout(SftpEditor.updateGutter, 50);

        } catch(e) {
            textarea.value = "Error loading file:\n" + e;
            statusMsg.innerText = "Error";
            statusMsg.style.color = "#ff5555";
            SftpEditor.updateGutter();
        }
    },

    save: async () => {
        if (!SftpEditor.currentFile) return;

        const textarea = document.getElementById('sftp-textarea');
        if (textarea.readOnly) {
            alert("Cannot save truncated file. Please download it instead.");
            return;
        }

        // BATCH 1 FIX: Capture Selection State
        const startPos = textarea.selectionStart;
        const endPos = textarea.selectionEnd;

        const statusMsg = document.getElementById('sftp-status-msg');
        const btn = document.getElementById('sftp-save-btn');
        
        const content = textarea.value;
        statusMsg.innerText = "Uploading...";
        btn.disabled = true;
        textarea.disabled = true;

        const hostId = SftpUtils.getActiveSshId();

        try {
            await invoke('sftp_write_file', {
                hostId: hostId,
                path: SftpEditor.currentFile,
                content: content
            });
            
            SftpEditor.originalContent = content;
            SftpEditor.setDirty(false);
            statusMsg.innerText = "Saved successfully";
            
            statusMsg.style.color = "#50fa7b";
            setTimeout(() => { statusMsg.style.color = "#888"; }, 2000);

            if (window.refreshSftp) {
                window.refreshSftp(undefined, false);
            }

        } catch (e) {
            alert("Upload Failed: " + e);
            statusMsg.innerText = "Upload Failed";
            statusMsg.style.color = "#ff5555";
        } finally {
            textarea.disabled = false;
            
            // BATCH 1 FIX: Restore Focus and Selection
            textarea.focus();
            textarea.setSelectionRange(startPos, endPos);
            
            if (textarea.value !== SftpEditor.originalContent) {
                SftpEditor.setDirty(true);
            } else {
                btn.disabled = true;
            }
        }
    }
};