// FILE: ui/js/sftp/inputs.js
// --------------------------

const SftpInputs = {
    isInputActive: false, // Guard flag

    // --- INLINE RENAME ---
    startInlineRename: (li, oldPath, onRenameCallback) => {
        const nameSpan = li.querySelector('.sftp-name');
        if (!nameSpan) return;

        // Clean name extraction
        const rawText = nameSpan.innerText;
        const iconMatch = rawText.match(/^[📁📄]/);
        const iconPart = iconMatch ? iconMatch[0] : ""; 
        const currentName = rawText.substring(iconPart.length + 1).trim();

        nameSpan.classList.add('renaming');
        nameSpan.parentElement.style.overflow = "visible";

        // Create elements programmatically to ensure cleaner event binding
        nameSpan.innerHTML = "";
        nameSpan.appendChild(document.createTextNode(iconPart + " "));
        
        const input = document.createElement('input');
        input.type = "text";
        input.className = "rename-input";
        input.value = currentName;
        input.setAttribute('draggable', 'false');
        nameSpan.appendChild(input);
        
        input.focus();
        SftpInputs.isInputActive = true;
        
        const dotIdx = currentName.lastIndexOf('.');
        if(dotIdx > 0) {
            input.setSelectionRange(0, dotIdx);
        } else {
            input.select();
        }

        let finished = false;

        const cleanup = () => {
            SftpInputs.isInputActive = false;
            nameSpan.classList.remove('renaming');
            nameSpan.parentElement.style.overflow = "hidden"; 
        };

        const restore = () => {
            cleanup();
            nameSpan.innerHTML = `${iconPart} ${currentName}`;
            nameSpan.title = currentName;
        };

        const commit = () => {
            if(finished) return;
            finished = true;
            
            cleanup();
            const newName = input.value.trim();
            
            if (newName && newName !== currentName) {
                console.log(`[Rename] Committing: ${oldPath} -> ${newName}`);
                // Try window.SftpOps first (most reliable)
                if(window.SftpOps && typeof window.SftpOps.renameEntry === 'function') {
                    window.SftpOps.renameEntry(oldPath, newName);
                } 
                // Fallback to callback if passed
                else if (typeof onRenameCallback === 'function') {
                    onRenameCallback(oldPath, newName);
                } else {
                    console.error("[Rename] No valid rename handler found!");
                    restore();
                }
            } else {
                restore();
            }
        };

        const cancel = () => {
            if(finished) return;
            finished = true;
            restore();
        };

        // --- Event Listeners ---

        // STOP Propagation of click to the LI row
        input.addEventListener('click', (e) => {
            e.stopPropagation();
            e.preventDefault();
        });
        
        input.addEventListener('mousedown', (e) => {
            e.stopPropagation();
        });

        input.addEventListener('keydown', (e) => {
            e.stopPropagation(); 
            if (e.key === 'Enter') { e.preventDefault(); commit(); }
            if (e.key === 'Escape') { e.preventDefault(); cancel(); }
        });

        input.addEventListener('blur', () => {
            // Delay to allow button clicks to register if necessary
            // But main issue was likely click propagation to LI
            setTimeout(commit, 150); 
        });
    },

    // --- PHANTOM INPUT (CREATE) ---
    showPhantomInput: (isDir, onCommit) => {
        const activeTab = typeof SftpTabManager !== 'undefined' ? SftpTabManager.getActive() : null;
        if (!activeTab || !activeTab.panel) return;
        
        const list = activeTab.panel.querySelector('.sftp-list');
        if(!list) return;

        const emptyState = list.querySelector('.empty-state');
        if(emptyState) emptyState.remove();

        const li = document.createElement('li');
        li.className = "phantom-row";
        li.style.padding = "2px 8px";
        
        const icon = isDir ? "📁" : "📄";
        
        const iconSpan = document.createElement('span');
        iconSpan.style.marginRight = "5px";
        iconSpan.innerText = icon;
        
        const input = document.createElement('input');
        input.type = "text";
        input.className = "phantom-input";
        input.placeholder = "Name...";
        
        li.appendChild(iconSpan);
        li.appendChild(input);
        
        const dotDot = list.querySelector('li');
        if (dotDot && dotDot.innerText.includes("..")) {
            dotDot.after(li);
        } else {
            list.prepend(li);
        }

        input.focus();
        SftpInputs.isInputActive = true;

        let finished = false;
        
        const cleanup = () => { 
            SftpInputs.isInputActive = false;
            if (li.parentNode) li.remove(); 
        };
        
        const commit = () => {
            if (finished) return;
            finished = true;
            const val = input.value.trim();
            cleanup();
            if(val) onCommit(val);
        };

        const cancel = () => {
            if (finished) return;
            finished = true;
            cleanup();
        };

        input.addEventListener('click', (e) => e.stopPropagation());
        input.addEventListener('mousedown', (e) => e.stopPropagation());

        input.addEventListener('keydown', (e) => {
            e.stopPropagation();
            if(e.key === 'Enter') { e.preventDefault(); commit(); }
            if(e.key === 'Escape') { e.preventDefault(); cancel(); }
        });
        
        input.addEventListener('blur', () => {
            setTimeout(commit, 150);
        });
    }
};