// FILE: ui/js/sftp/main.js
// ------------------------
// Entry point for SFTP interactions & Global Keys

function initSFTP() {
    SftpEditor.init();
    
    document.addEventListener('click', (e) => {
        if (!e.target.closest('.ctx-menu')) SftpUI.hideContextMenu();
    });

    document.addEventListener('keydown', (e) => {
        const sftpView = document.getElementById('view-sftp');
        if (!sftpView || !sftpView.classList.contains('active')) return;

        // F6: Toggle Focus between Editor and File List
        if (e.key === 'F6') {
            e.preventDefault();
            const listEl = SftpUI.getActiveList();
            const editorEl = document.getElementById('sftp-textarea');
            
            if (document.activeElement === editorEl) {
                if (listEl) {
                    listEl.focus();
                    // Highlight the last known item or first item
                    SftpUI.restoreFocus();
                }
            } else {
                if (editorEl && !editorEl.disabled) editorEl.focus();
            }
            return;
        }

        const activeTab = typeof SftpTabManager !== 'undefined' ? SftpTabManager.getActive() : null;
        if (!activeTab) return;

        const list = activeTab.panel.querySelector('.sftp-list');
        const activeEl = document.activeElement;
        
        // Don't trigger hotkeys if typing in inputs (except for F6 above)
        if (activeEl.tagName === 'INPUT' || activeEl.tagName === 'TEXTAREA') return;

        const isSidebar = activeEl === list || activeTab.panel.contains(activeEl);

        if (e.key === 'Escape') SftpUI.hideContextMenu();

        // Keyboard Navigation (Arrow Keys)
        if (isSidebar) {
            if (e.key === 'ArrowDown') {
                e.preventDefault();
                SftpUI.moveFocus(1);
            } else if (e.key === 'ArrowUp') {
                e.preventDefault();
                SftpUI.moveFocus(-1);
            } else if (e.key === 'Enter') {
                e.preventDefault();
                const focused = SftpUI.getFocusedItem();
                if (focused) focused.click();
            } else if (e.key === 'Backspace') {
                e.preventDefault();
                // Navigate Up
                const parent = SftpUtils.getParentPath(activeTab.path);
                if (parent && window.refreshSftp) window.refreshSftp(parent);
            }
        }

        if (e.key === 'Delete' && isSidebar) {
            e.preventDefault();
            const selectedItems = SftpUI.getSelectedItems();
            if (selectedItems.length > 0) {
                if (selectedItems.length > 1) {
                    selectedItems.forEach(i => SftpOps.deleteEntry(i.path, i.isDir, false));
                } else {
                    const item = selectedItems[0];
                    if (!item.name.includes("..")) {
                        SftpOps.deleteEntry(item.path, item.isDir, false);
                    }
                }
            }
        }
        
        if (e.key === 'F2' && isSidebar) {
            e.preventDefault();
            const listEl = SftpUI.getActiveList();
            if (!listEl) return;
            
            const selectedRows = listEl.querySelectorAll('li.selected');
            if (selectedRows.length === 1) {
                const li = selectedRows[0];
                const path = li.dataset.path;
                if (!li.innerText.includes("..")) {
                    SftpUI.startInlineRename(li, path);
                }
            }
        }
        
        if ((e.ctrlKey || e.metaKey) && e.key.toLowerCase() === 'v' && isSidebar) {
            e.preventDefault();
            SftpClipboard.paste(activeTab.path);
        }

        if ((e.ctrlKey || e.metaKey) && e.shiftKey) {
            if (e.key.toLowerCase() === 'n') { 
                e.preventDefault(); 
                SftpOps.createEntry(true); 
            }
            if (e.key.toLowerCase() === 'j') { 
                e.preventDefault(); 
                SftpOps.createEntry(false); 
            }
        }
    });

    window.refreshSftp = refreshSftp;
}

// ... refreshSftp remains unchanged ...
async function refreshSftp(pathOverride, recordHistory = true) {
    if (typeof SftpTabManager === 'undefined') return;
    
    const activeTab = SftpTabManager.getActive();
    if (!activeTab) return; 

    if (!activeTab.panel || !activeTab.panel.querySelector('.sftp-list')) {
        setTimeout(() => refreshSftp(pathOverride, recordHistory), 100);
        return;
    }

    const path = pathOverride || activeTab.path || ".";
    activeTab.path = path; 

    const input = activeTab.panel.querySelector('.sftp-path-input');
    if (input && document.activeElement !== input) input.value = path;

    const hostId = activeTab.hostId;
    const listEl = activeTab.panel.querySelector('.sftp-list');
    
    const callbacks = {
        onNavigate: (newPath) => refreshSftp(newPath, true),
        onOpen: (filePath, size) => SftpEditor.load(filePath, size),
        onDelete: (filePath, isDir, force) => SftpOps.deleteEntry(filePath, isDir, force),
        onMove: (src, dst) => SftpOps.moveEntry(src, dst),
        onUpload: (fileList, dst) => SftpTransfer.uploadFiles(fileList, dst)
    };

    const handleResult = (files) => {
         SftpUI.renderList(listEl, files, path, callbacks);
    };

    const handleError = (e) => {
        SftpUI.showError(e);
        const errStr = String(e).toLowerCase();
        if (errStr.includes("permission") || errStr.includes("access")) {
            SftpOps.diagnoseError("Folder Access", path, e);
        }
    };

    if (hostId === "LOCAL") {
        SftpUI.showLoading();
        try {
            const files = await invoke('sftp_list_files', { hostId: "LOCAL", path });
            handleResult(files);
        } catch(e) { handleError(e); }
        return;
    }

    if (!hostId || hostId === "quick_connect") {
        SftpUI.showNotConnected();
        return;
    }

    SftpUI.showLoading();

    try {
        const files = await invoke('sftp_list_files', { hostId, path });
        handleResult(files);
    } catch(e) { handleError(e); }
}