// ui/js/sftp/session.js

const SftpSession = {
    sessions: {},
    activeId: null,

    init: () => {
        console.log("[Session] Init. Creating default.");
        SftpSession.create("Local Files", "local", "default");
    },

    create: async (name, hostId, forcedSessionId = null) => {
        const sessionId = forcedSessionId || "sftp_" + Math.random().toString(36).substr(2, 9);
        console.log(`[Session] Creating ${sessionId} (${name}, ${hostId})`);
        
        SftpSession.sessions[sessionId] = {
            id: sessionId,
            name: name,
            hostId: hostId,
            path: ".",
            history: [],
            historyPtr: -1,
            fileCache: [], 
            scrollPos: 0
        };

        const container = document.getElementById('sftp-tabs-container');
        if (container) {
            const div = document.createElement('div');
            div.className = 'sftp-tab';
            div.dataset.id = sessionId;
            div.innerHTML = `
                <span class="tab-name">${name}</span>
                <div class="sftp-tab-close">×</div>
            `;
            div.onclick = () => SftpSession.activate(sessionId);
            div.querySelector('.sftp-tab-close').onclick = (e) => {
                e.stopPropagation();
                SftpSession.close(sessionId);
            };
            container.appendChild(div);
        } else {
            console.error("[Session] Tabs container missing in DOM");
        }

        if (hostId) {
            try {
                console.log(`[Session] Registering backend: ${sessionId}`);
                await invoke('sftp_open_session', { sessionId, hostId });
            } catch (e) { 
                console.error("[Session] Backend Open Error:", e); 
            }
        }

        SftpSession.activate(sessionId);
        SftpSession.renderShelf();
    },

    activate: (sessionId) => {
        if (!SftpSession.sessions[sessionId]) return;
        
        SftpSession.activeId = sessionId;
        document.querySelectorAll('.sftp-tab').forEach(t => {
            t.classList.toggle('active', t.dataset.id === sessionId);
        });

        const s = SftpSession.sessions[sessionId];

        SftpUI.updatePathInput(s.path);
        
        if (s.fileCache && s.fileCache.length > 0) {
            SftpUI.renderList(s.fileCache, s.path, 
                (newPath) => window.refreshSftp(newPath),
                (f, size) => SftpEditor.load(f, size),
                (f, isDir, force) => SftpOps.deleteEntry(f, isDir, force),
                (src, dst) => SftpOps.moveEntry(src, dst),
                (list, dst) => SftpTransfer.uploadFiles(list, dst)
            );
            const listEl = document.getElementById('sftp-list');
            if(listEl) listEl.scrollTop = s.scrollPos || 0;
        } else {
            window.refreshSftp(s.path);
        }
        
        SftpSession.renderShelf(); 
    },

    close: async (sessionId) => {
        const s = SftpSession.sessions[sessionId];
        if (!s) return;

        try { await invoke('sftp_close_session', { sessionId }); } catch(e) {}

        const tab = document.querySelector(`.sftp-tab[data-id="${sessionId}"]`);
        if (tab) tab.remove();
        
        delete SftpSession.sessions[sessionId];

        if (SftpSession.activeId === sessionId) {
            const keys = Object.keys(SftpSession.sessions);
            if (keys.length > 0) {
                SftpSession.activate(keys[keys.length - 1]);
            } else {
                SftpSession.create("Local Files", "local", "default");
            }
        }
        SftpSession.renderShelf();
    },

    getActive: () => SftpSession.sessions[SftpSession.activeId],

    updateState: (sessionId, path, files) => {
        if (SftpSession.sessions[sessionId]) {
            SftpSession.sessions[sessionId].path = path;
            if (files) SftpSession.sessions[sessionId].fileCache = files;
        }
    },
    
    saveScroll: () => {
        if (SftpSession.activeId && SftpSession.sessions[SftpSession.activeId]) {
            const listEl = document.getElementById('sftp-list');
            if(listEl) SftpSession.sessions[SftpSession.activeId].scrollPos = listEl.scrollTop;
        }
    },

    renderShelf: () => {
        const shelfList = document.getElementById('sftp-connections-list');
        if (!shelfList) {
            console.warn("[Session] Shelf list container missing");
            return;
        }

        shelfList.innerHTML = '';
        const keys = Object.keys(SftpSession.sessions);
        console.log(`[Session] Rendering shelf. Count: ${keys.length}`);

        if (keys.length === 0) {
            shelfList.innerHTML = '<div class="shelf-empty">No active sessions.</div>';
            return;
        }

        keys.forEach(id => {
            const s = SftpSession.sessions[id];
            const row = document.createElement('div');
            row.style.display = "flex";
            row.style.alignItems = "center";
            row.style.justifyContent = "space-between";
            row.style.padding = "4px 8px";
            row.style.borderBottom = "1px solid #333";
            row.style.fontSize = "12px";
            row.style.backgroundColor = (id === SftpSession.activeId) ? "#2d2a3d" : "transparent";
            
            const info = document.createElement('span');
            info.style.cursor = "pointer";
            info.style.flexGrow = "1";
            info.innerHTML = `<strong>${s.name}</strong> <span style="color:#888">(${s.hostId})</span> - ${s.path}`;
            info.onclick = () => SftpSession.activate(id);

            const closeBtn = document.createElement('button');
            closeBtn.innerText = "✖";
            closeBtn.className = "del-btn";
            closeBtn.style.color = "#ccc";
            closeBtn.onclick = (e) => { e.stopPropagation(); SftpSession.close(id); };

            row.appendChild(info);
            row.appendChild(closeBtn);
            shelfList.appendChild(row);
        });
    }
};