// ui/js/sftp/transfer.js
// Handles File Uploads with Auto-Retry and Concurrency Management

const SftpTransfer = {
    isCancelled: false,

    uploadFiles: async (files, destFolder) => {
        const hostId = SftpUtils.getActiveSshId();
        if (!hostId) return;

        SftpTransfer.isCancelled = false;

        // --- UI Setup ---
        const progressModal = document.getElementById('progress-modal');
        const progressTitle = document.getElementById('progress-title');
        const progressDetail = document.getElementById('progress-detail');
        const progressBar = document.getElementById('progress-bar-fill');
        const cancelBtn = document.getElementById('progress-cancel-btn');

        const showProgress = (show) => {
            if (progressModal) progressModal.style.display = show ? 'flex' : 'none';
        };

        if (cancelBtn) {
            cancelBtn.style.display = "inline-block";
            cancelBtn.onclick = async () => {
                SftpTransfer.isCancelled = true;
                progressTitle.innerText = "Cancelling...";
                progressDetail.innerText = "Please wait...";
                try {
                    await invoke('cancel_sftp_transfer', { hostId: hostId });
                } catch (e) { console.error("Cancel invoke failed:", e); }
            };
        }

        showProgress(true);

        // --- Core Worker Logic ---
        const processBatch = async (batchFiles, concurrency, contextPrefix = "") => {
            let completedCount = 0;
            let failedFiles = [];
            const totalCount = batchFiles.length;
            const queue = [...batchFiles];

            const uploadWorker = async () => {
                while (queue.length > 0) {
                    if (SftpTransfer.isCancelled) break;

                    const file = queue.shift();

                    // Update UI
                    if (progressModal) {
                        const pct = (((completedCount) / totalCount) * 100).toFixed(0);
                        progressTitle.innerHTML = `${contextPrefix} ${completedCount}/${totalCount}`;
                        progressDetail.innerText = file.name;
                        progressBar.style.width = pct + "%";
                        // Visual cue for retry mode
                        progressBar.style.backgroundColor = (concurrency < 10) ? "#f1fa8c" : "#007acc"; 
                    }

                    try {
                        let targetDir = destFolder;
                        if (file.uploadRelPath) {
                            targetDir = SftpUtils.joinPath(destFolder, file.uploadRelPath);
                            // Ensure folder exists (best effort, swallow error if exists)
                            try {
                                if (!SftpTransfer.isCancelled) {
                                    await invoke('sftp_create_entry', { hostId, path: targetDir, isDir: true });
                                }
                            } catch (e) { /* Ignore folder creation errors, might already exist */ }
                        }

                        const remotePath = SftpUtils.joinPath(targetDir, file.name);
                        
                        // Read fresh every time (in case of retry)
                        const base64 = await SftpTransfer.readFileAsBase64(file);

                        if (SftpTransfer.isCancelled) break;

                        await invoke('sftp_write_binary', {
                            hostId: hostId,
                            path: remotePath,
                            base64Data: base64
                        });

                    } catch (e) {
                        if (String(e).includes("Cancelled")) {
                            SftpTransfer.isCancelled = true;
                        } else {
                            console.error(`Upload Error (${file.name}):`, e);
                            failedFiles.push(file); // Re-queue for next pass
                        }
                    } finally {
                        completedCount++;
                    }
                }
            };

            const workers = [];
            const poolSize = Math.min(totalCount, concurrency);
            for (let i = 0; i < poolSize; i++) {
                workers.push(uploadWorker());
            }

            await Promise.all(workers);
            return failedFiles;
        };

        // --- Phase 1: Initial High-Speed Run ---
        let pendingFiles = [...files];
        
        progressTitle.innerText = "Starting Upload...";
        pendingFiles = await processBatch(pendingFiles, 10, "Uploading");

        // --- Phase 2: Auto-Retry (Once, Low Concurrency) ---
        if (!SftpTransfer.isCancelled && pendingFiles.length > 0) {
            console.log(`[SFTP] Auto-retrying ${pendingFiles.length} files...`);
            // Brief pause to let sockets drain/reset if network glitched
            await new Promise(r => setTimeout(r, 500)); 
            
            pendingFiles = await processBatch(pendingFiles, 4, "Auto-Retrying");
        }

        // --- Phase 3: Manual Retry Loop ---
        while (!SftpTransfer.isCancelled && pendingFiles.length > 0) {
            showProgress(false); // Hide progress bar to show modal
            
            const msg = `Upload finished with ${pendingFiles.length} errors.\n\nWould you like to retry the failed files?`;
            const shouldRetry = await Modal.confirm(msg, "Upload Incomplete");
            
            if (shouldRetry) {
                showProgress(true);
                // Keep concurrency low for manual retries too, implies difficult network
                pendingFiles = await processBatch(pendingFiles, 4, "Retrying"); 
            } else {
                break; // User Abort
            }
        }

        // --- Final UI Cleanup ---
        if (progressModal && progressModal.style.display !== 'none') {
            if (SftpTransfer.isCancelled) {
                progressTitle.innerText = "Upload Cancelled";
                progressDetail.innerText = "Stopped by user.";
                progressBar.style.backgroundColor = "#ff5555";
            } else if (pendingFiles.length === 0) {
                progressTitle.innerText = "Upload Complete";
                progressBar.style.width = "100%";
                progressBar.style.backgroundColor = "#50fa7b"; // Green
                progressDetail.innerText = "Success.";
            } else {
                progressTitle.innerText = "Upload Finished";
                progressDetail.innerText = `${pendingFiles.length} files failed.`;
                progressBar.style.backgroundColor = "#ff5555";
            }

            if (cancelBtn) cancelBtn.style.display = "none";

            setTimeout(() => {
                showProgress(false);
                progressBar.style.width = "0%";
                progressBar.style.backgroundColor = "#007acc";
                SftpTransfer.isCancelled = false;
            }, 2500);
        }

        if (window.refreshSftp) window.refreshSftp();
    },

    readFileAsBase64: (file) => {
        return new Promise((resolve, reject) => {
            const reader = new FileReader();
            reader.onload = () => {
                const result = reader.result;
                const b64 = result.split(',')[1];
                resolve(b64);
            };
            reader.onerror = reject;
            reader.readAsDataURL(file);
        });
    }
};

window.uploadFiles = SftpTransfer.uploadFiles;