// FILE: ui/js/sftp/ui.js
// ----------------------
// DOM helpers for SFTP

const SftpUI = {
    dragState: null,
    isNavigating: false,
    selection: new Set(),
    lastFocusedPath: null,

    getActiveList: () => {
        if (typeof SftpTabManager === 'undefined') return document.getElementById('sftp-list');
        const tab = SftpTabManager.getActive();
        if (!tab || !tab.panel) return null;
        return tab.panel.querySelector('.sftp-list');
    },

    // --- KEYBOARD NAVIGATION HELPERS ---

    getFocusedItem: () => {
        const list = SftpUI.getActiveList();
        if (!list) return null;
        return list.querySelector('li.focused-file');
    },

    restoreFocus: () => {
        const list = SftpUI.getActiveList();
        if (!list) return;
        
        const items = Array.from(list.querySelectorAll('li:not(.empty-state)'));
        if (items.length === 0) return;

        // Try to find last focused
        if (SftpUI.lastFocusedPath) {
            const el = items.find(li => li.dataset.path === SftpUI.lastFocusedPath);
            if (el) {
                SftpUI.setFocusVisuals(el);
                return;
            }
        }
        
        // Default to first item (likely "..")
        SftpUI.setFocusVisuals(items[0]);
    },

    moveFocus: (direction) => {
        const list = SftpUI.getActiveList();
        if (!list) return;

        const items = Array.from(list.querySelectorAll('li:not(.empty-state)'));
        if (items.length === 0) return;

        let currentIndex = -1;
        if (SftpUI.lastFocusedPath) {
            currentIndex = items.findIndex(li => li.dataset.path === SftpUI.lastFocusedPath);
        }

        if (currentIndex === -1) {
            // Start at top if nothing focused
            currentIndex = 0;
        } else {
            currentIndex += direction;
        }

        // Clamp
        if (currentIndex < 0) currentIndex = 0;
        if (currentIndex >= items.length) currentIndex = items.length - 1;

        const target = items[currentIndex];
        
        // Update Visuals
        SftpUI.setFocusVisuals(target);
        
        // Update Selection to follow focus (Windows Explorer style default)
        SftpUI.setSelection(target.dataset.path);

        // Scroll into view
        target.scrollIntoView({ block: 'nearest' });
    },

    setFocusVisuals: (li) => {
        if (!li) return;
        const list = SftpUI.getActiveList();
        list.querySelectorAll('li').forEach(el => el.classList.remove('focused-file'));
        li.classList.add('focused-file');
        SftpUI.lastFocusedPath = li.dataset.path;
    },

    // --- EXISTING METHODS ---

    showLoading: () => {
        const list = SftpUI.getActiveList();
        SftpUI.clearSelection();
        if (list) list.innerHTML = '<li class="empty-state">Loading...</li>';
    },

    showError: (msg) => {
        const list = SftpUI.getActiveList();
        const status = document.getElementById('sftp-status-msg');
        
        console.error("SFTP Error:", msg);
        if(status) { 
            status.innerText = "Error: " + msg; 
            status.style.color = "red"; 
        }

        if (list) {
            list.innerHTML = `<li class="empty-state" style="color:red">Error: ${msg}</li>`;
            const tab = typeof SftpTabManager !== 'undefined' ? SftpTabManager.getActive() : null;
            if (tab && tab.path !== "/" && tab.path !== ".") {
                const li = document.createElement('li');
                li.innerHTML = `<span class="sftp-name text-accent">📁 .. (Go Up)</span>`;
                li.style.borderTop = "1px solid #3e3e42";
                li.style.marginTop = "5px";
                li.onclick = () => {
                    const parent = SftpUtils.getParentPath(tab.path);
                    if (parent && window.refreshSftp) window.refreshSftp(parent);
                };
                list.appendChild(li);
            }
        }
        SftpUI.isNavigating = false;
    },

    showNotConnected: () => {
        const list = SftpUI.getActiveList();
        if (list) list.innerHTML = '<li class="empty-state" style="color:#e6a23c">Not Connected / Select a Tab</li>';
        SftpUI.isNavigating = false;
    },

    addToSelection: (path) => {
        SftpUI.selection.add(path);
        SftpUI.redrawSelection();
    },

    removeFromSelection: (path) => {
        SftpUI.selection.delete(path);
        SftpUI.redrawSelection();
    },

    setSelection: (pathOrArray) => {
        SftpUI.selection.clear();
        if (Array.isArray(pathOrArray)) {
            pathOrArray.forEach(p => SftpUI.selection.add(p));
        } else if (pathOrArray) {
            SftpUI.selection.add(pathOrArray);
        }
        SftpUI.redrawSelection();
    },

    clearSelection: () => {
        SftpUI.selection.clear();
        SftpUI.redrawSelection();
    },

    redrawSelection: () => {
        const list = SftpUI.getActiveList();
        if(!list) return;
        
        const items = list.querySelectorAll('li');
        items.forEach(li => {
            const path = li.dataset.path;
            if (SftpUI.selection.has(path)) {
                li.classList.add('selected');
            } else {
                li.classList.remove('selected');
            }
            if (path === SftpUI.lastFocusedPath) {
                li.classList.add('focused-file');
            } else {
                li.classList.remove('focused-file');
            }
        });
    },

    getSelectedItems: () => {
        const list = SftpUI.getActiveList();
        if(!list) return [];
        
        const results = [];
        const items = list.querySelectorAll('li');
        
        items.forEach(li => {
            if (li.classList.contains('selected')) {
                // Fix: Get clean name from dataset path to avoid icons
                // Note: We use the path basename
                const path = li.dataset.path;
                let cleanName = path;
                
                // Simple basename extraction
                if (path.includes('/')) cleanName = path.split('/').pop();
                else if (path.includes('\\')) cleanName = path.split('\\').pop();

                results.push({
                    path: path,
                    name: cleanName,
                    isDir: li.dataset.isdir === "true"
                });
            }
        });
        return results;
    },

    renderList: (listEl, files, currentPath, callbacks) => {
        if (!listEl) return;
        SftpUI.isNavigating = false;
        SftpUI.selection.clear();
        
        // Preserve focus if reloading same path and user hasn't moved
        // otherwise reset.
        if (SftpRender.currentPathRef !== currentPath) {
             SftpUI.lastFocusedPath = null;
        }

        SftpRender.list(listEl, files, currentPath, {
            onNavigate: (newPath) => {
                if(SftpUI.isNavigating) return; 
                SftpUI.isNavigating = true;
                callbacks.onNavigate(newPath);
            },
            onOpen: callbacks.onOpen,
            onDelete: callbacks.onDelete,
            
            onDragStart: (path, name) => { 
                if (SftpUI.selection.has(path)) {
                    const items = SftpUI.getSelectedItems();
                    SftpUI.dragState = { mode: 'multi', items: items };
                } else {
                    SftpUI.dragState = { mode: 'single', path: path, name: name };
                }
            },
            
            onDropInternal: async (destFolder) => {
                if (!SftpUI.dragState) return;
                
                if (SftpUI.dragState.mode === 'multi') {
                    for (const item of SftpUI.dragState.items) {
                        if (item.path === destFolder) continue; 
                        const destPath = SftpUtils.joinPath(destFolder, item.name);
                        await callbacks.onMove(item.path, destPath);
                    }
                } else {
                    const src = SftpUI.dragState.path;
                    const destPath = SftpUtils.joinPath(destFolder, SftpUI.dragState.name);
                    callbacks.onMove(src, destPath);
                }
                SftpUI.dragState = null;
            },
            
            onDropExternal: (fileList, destFolder) => {
                const files = Array.from(fileList);
                callbacks.onUpload(files, destFolder);
            },
            onContextMenu: (x, y, fileData) => {
                if (!SftpUI.selection.has(fileData.path)) {
                    SftpUI.setSelection(fileData.path);
                    SftpUI.lastFocusedPath = fileData.path;
                }
                const selectedItems = SftpUI.getSelectedItems();
                SftpMenus.show(x, y, selectedItems, {
                    onRename: (fd) => {
                        const items = listEl.querySelectorAll('li');
                        let targetLi = null;
                        items.forEach(li => { if(li.dataset.path === fd.path) targetLi = li; });
                        if(targetLi) SftpInputs.startInlineRename(targetLi, fd.path, window.renameEntry);
                    },
                    onDelete: (items) => {
                        items.forEach(i => callbacks.onDelete(i.path, i.isDir, false));
                    },
                    onCopy: (items) => SftpClipboard.copy(items),
                    onCut: (items) => SftpClipboard.cut(items)
                });
            }
        });
        
        // If we have a last focused path, restore visuals
        if (SftpUI.lastFocusedPath) {
             setTimeout(SftpUI.restoreFocus, 0);
        }
    },

    hideContextMenu: () => SftpMenus.hide(),
    startInlineRename: (li, oldPath) => SftpInputs.startInlineRename(li, oldPath, window.renameEntry),
    showPhantomInput: (isDir, onCommit) => SftpInputs.showPhantomInput(isDir, onCommit),
    
    updatePathInput: (path) => {}
};