// FILE: ui/js/sftp/utils.js
// -------------------------

const SftpUtils = {
    // Batch 7 Polish: Robust Path Joining
    joinPath: (base, segment) => {
        // Force forward slashes
        const cleanBase = base.replace(/\\/g, '/').replace(/\/$/, '');
        const cleanSeg = segment.replace(/\\/g, '/').replace(/^\//, '');
        
        if (cleanBase === '' || cleanBase === '.') return cleanSeg;
        if (cleanSeg === '' || cleanSeg === '.') return cleanBase;
        
        return `${cleanBase}/${cleanSeg}`;
    },

    getParentPath: (path) => {
        if (!path || path === "/") return null;
        if (path === ".") return "..";
        if (path === "..") return "../..";
        
        const cleanPath = path.replace(/\\/g, '/');
        const parts = cleanPath.split('/').filter(p => p.length > 0);
        parts.pop(); 

        if (parts.length === 0) return cleanPath.startsWith("/") ? "/" : ".";
        const newPath = parts.join('/');
        return cleanPath.startsWith("/") ? "/" + newPath : newPath;
    },

    getActiveSshId: () => {
        if (typeof SftpTabManager !== 'undefined') {
            const tab = SftpTabManager.getActive();
            if (tab && tab.hostId) {
                // If it's a "Local" tab, return special ID
                if (tab.hostId === "LOCAL") return "LOCAL";
                return tab.hostId;
            }
        }
        
        // Fallback to active SSH session if SFTP tab manager isn't ready (rare)
        const activeSid = AppState.ssh.activeSessionId;
        if (activeSid) {
            const session = AppState.ssh.sessions[activeSid];
            // If quick connect, return the session ID itself
            if (session && session.hostId === "quick_connect") return activeSid; 
            if (session) return session.hostId;
        }
        
        return null;
    },

    formatShortSize: (bytes) => {
        if (bytes === 0) return "0B";
        const k = 1024;
        const sizes = ['B', 'K', 'M', 'G', 'T'];
        const i = Math.floor(Math.log(bytes) / Math.log(k));
        let val = (bytes / Math.pow(k, i));
        let decimals = (val < 10 && i > 0) ? 1 : 0; 
        return parseFloat(val.toFixed(decimals)) + sizes[i];
    },

    // Debounce Helper
    debounce: (func, wait) => {
        let timeout;
        return function(...args) {
            clearTimeout(timeout);
            timeout = setTimeout(() => func.apply(this, args), wait);
        };
    }
};