// FILE: ui/js/ssh/session.js
// --------------------------

const SshSession = {
    
    init: () => {
        window.addEventListener('resize', SshSession.debounce(() => {
            const sid = AppState.ssh.activeSessionId;
            if (sid) SshSession.fitAndSend(sid);
        }, 200));
    },

    create: async (name, hostId, payload) => {
        const sessionId = "sess_" + Math.random().toString(36).substr(2, 9);
        const termsContainer = document.getElementById('ssh-terminals-container');

        const termDiv = document.createElement('div');
        termDiv.className = 'terminal-instance';
        termDiv.id = `term-${sessionId}`;
        termsContainer.appendChild(termDiv);

        const term = new Terminal({
            fontFamily: 'Consolas, monospace',
            fontSize: 14,
            wordWrap: true,
            allowProposedApi: true,
            drawBoldTextInBrightColors: true,
            theme: {
                background: '#1e1e1e', foreground: '#d4d4d4', cursor: '#ffffff',
                selection: 'rgba(255, 255, 255, 0.2)',
                black: '#000000', red: '#cd3131', green: '#0dbc79', yellow: '#e5e510',
                blue: '#2472c8', magenta: '#bc3fbc', cyan: '#11a8cd', white: '#e5e5e5',
                brightBlack: '#666666', brightRed: '#f14c4c', brightGreen: '#23d18b',
                brightYellow: '#f5f543', brightBlue: '#3b8eea', brightMagenta: '#d670d6',
                brightCyan: '#29b8db', brightWhite: '#ffffff'
            }
        });

        const fit = new FitAddon.FitAddon();
        term.loadAddon(fit);
        term.open(termDiv);
        fit.fit();

        term.onData(async (data) => {
            const cleanCheck = data.replace(/\x1b\[200~/g, "").replace(/\x1b\[201~/g, "");
            const hasNewline = cleanCheck.includes('\r') || cleanCheck.includes('\n');
            if (hasNewline && cleanCheck.length > 1) {
                const lines = cleanCheck.split(/\r?\n|\r/).filter(x => x.length > 0);
                if (lines.length > 1) {
                    const msg = `⚠️ PASTE WARNING ⚠️\n\nInput contains ${lines.length} lines.\n\nFirst line: "${lines[0].substring(0, 30)}..."\n\nSend to terminal?`;
                    const allowed = await Modal.confirm(msg, "Paste Safety");
                    if (!allowed) return;
                }
            }
            
            if (hostId === "LOCAL") {
                invoke('send_local_pty_data', { sessionId: sessionId, data: data }).catch(()=>{});
            } else {
                invoke('send_ssh_data', { sessionId: sessionId, data: data }).catch(()=>{});
            }
        });

        Utils.enableMobaStyleControls(term);

        AppState.ssh.sessions[sessionId] = {
            id: sessionId, name, term, fit, container: termDiv, hostId
        };

        if (typeof Connections !== 'undefined' && hostId !== "LOCAL") {
            Connections.add(sessionId, name, 'ssh', hostId, payload);
        }

        SshSession.activate(sessionId);
        
        if (hostId === "LOCAL") {
            term.writeln(`\x1b[36m--- Local Terminal ---\x1b[0m`);
            try {
                const cols = term.cols;
                const rows = term.rows;
                await invoke('spawn_local_pty', { sessionId, rows, cols });
            } catch (e) {
                term.writeln(`\r\n\x1b[31mFailed to spawn PTY: ${e}\x1b[0m`);
            }
        } else {
            term.writeln(`Connecting to ${name}...`);
            try {
                await invoke('connect_ssh', {
                    sessionId: sessionId,
                    hostId: hostId,
                    payload: payload
                });
                term.writeln(`\r\n\x1b[32mAuthenticated.\x1b[0m\r\n`);
                setTimeout(() => SshSession.fitAndSend(sessionId), 100);
                if (typeof Connections !== 'undefined') Connections.updateStatus(sessionId, 'Connected');
            } catch (e) {
                term.writeln(`\r\n\x1b[31mConnection Error: ${e}\x1b[0m`);
                if (typeof Connections !== 'undefined') Connections.updateStatus(sessionId, 'Error');
            }
        }
    },

    activate: (sessionId) => {
        AppState.ssh.activeSessionId = sessionId;
        
        Object.values(AppState.ssh.sessions).forEach(s => {
            if(s.container) s.container.classList.remove('active');
        });
        
        const s = AppState.ssh.sessions[sessionId];
        if(s) {
            if(s.container) s.container.classList.add('active');
            setTimeout(() => { 
                if(s.fit) s.fit.fit(); 
                if(s.term) s.term.focus(); 
            }, 50);
            
            const statusText = document.getElementById('ssh-status-text');
            if(statusText) statusText.innerText = "Active: " + s.name;
            
            const metrics = document.getElementById('ssh-metrics-container');
            if(metrics) metrics.style.display = (s.hostId === "LOCAL") ? 'none' : 'flex';
            
            if (typeof Connections !== 'undefined') Connections.render();
        }
    },

    // FIX: Removed Confirmation Dialog - Instant Close
    close: async (sessionId, force) => {
        const s = AppState.ssh.sessions[sessionId];
        if (!s) return;

        if (s.hostId === "LOCAL") {
            // Cleanup local logic if needed
        } else {
            await invoke('disconnect_ssh', { sessionId: sessionId });
        }
        
        if(s.container) s.container.remove(); 
        if(s.term) s.term.dispose();
        delete AppState.ssh.sessions[sessionId];
        
        if (typeof Connections !== 'undefined' && s.hostId !== "LOCAL") {
            Connections.remove(sessionId);
        }
        
        if(AppState.ssh.activeSessionId === sessionId) {
            const keys = Object.keys(AppState.ssh.sessions);
            if(keys.length > 0) SshSession.activate(keys[keys.length - 1]);
            else {
                AppState.ssh.activeSessionId = null;
                const st = document.getElementById('ssh-status-text');
                if(st) st.innerText = "SSH Idle";
                const mc = document.getElementById('ssh-metrics-container');
                if(mc) mc.style.display = 'none';
                
                if (typeof Connections !== 'undefined') Connections.render();
            }
        }
    },

    fitAndSend: (sid) => {
        const s = AppState.ssh.sessions[sid];
        if (s && s.fit) {
            s.fit.fit();
            const dims = s.fit.proposeDimensions();
            if (dims && dims.cols && dims.rows) {
                if (s.hostId === "LOCAL") {
                    invoke('resize_local_pty', { sessionId: sid, rows: dims.rows, cols: dims.cols });
                } else {
                    invoke('resize_ssh_session', { sessionId: sid, rows: dims.rows, cols: dims.cols });
                }
            }
        }
    },

    debounce: (func, wait) => {
        let timeout;
        return function(...args) {
            clearTimeout(timeout);
            timeout = setTimeout(() => func.apply(this, args), wait);
        };
    }
};