// FILE: ui/js/state.js
// --------------------

// Explicitly define global accessors
window.invoke = window.__TAURI__.core.invoke;
window.listen = window.__TAURI__.event.listen;

// Global State
window.AppState = {
    serial: { term: null, fit: null, activeSessionId: null },
    ssh: { sessions: {}, activeSessionId: null },
    
    // Shelf State
    shelf: {
        open: true,
        connections: [] 
    },
    
    log: (msg) => {
        const timestamp = new Date().toLocaleTimeString();
        console.log(`[${timestamp}] ${msg}`);
    }
};

let isPolling = true;
let lastMetrics = null;

async function startPoller() {
    pollSerialLoop();
    await initSshListeners();
    initShelf();
}

function initShelf() {
    const header = document.getElementById('shelf-header');
    if (header) {
        header.onclick = (e) => {
            if (e.target.closest('.shelf-tools')) return;
            toggleShelf();
        };
    }
}

function toggleShelf(forceState) {
    const shelf = document.getElementById('bottom-shelf');
    if (!shelf) return;

    if (forceState !== undefined) {
        window.AppState.shelf.open = forceState;
    } else {
        window.AppState.shelf.open = !window.AppState.shelf.open;
    }

    if (window.AppState.shelf.open) {
        shelf.classList.remove('shelf-collapsed');
        shelf.classList.add('shelf-expanded');
    } else {
        shelf.classList.remove('shelf-expanded');
        shelf.classList.add('shelf-collapsed');
    }

    // Give transition time to finish before firing resize for xterm fit
    setTimeout(() => {
        window.dispatchEvent(new Event('resize'));
    }, 320);
}

async function initSshListeners() {
    await window.listen('ssh-data', e => {
        const { id, data } = e.payload;
        const session = window.AppState.ssh.sessions[id];
        if (session && session.term) {
            session.term.write(new Uint8Array(data));
        }
    });

    await window.listen('ssh-metrics', e => {
        const { id, stats } = e.payload;
        if (id === window.AppState.ssh.activeSessionId) {
            updateMetrics(stats);
        }
    });
    
    window.AppState.log("SSH Event Listeners Initialized");
}

async function pollSerialLoop() {
    if (!isPolling) return;
    
    if (window.AppState.serial.activeSessionId) {
        try {
            const res = await window.invoke('poll_serial', { sessionId: window.AppState.serial.activeSessionId });
            
            // 1. Data Pipe
            if (res.data && window.AppState.serial.term) {
                window.AppState.serial.term.write(Utils.base64ToUint8Array(res.data));
            }
            
            // 2. Status Update (Batch 3: Visual Feedback)
            if (res.status) {
                // If the Serial object is available (loaded from serial.js), use its helper
                if (typeof Serial !== 'undefined' && typeof Serial.updateStatus === 'function') {
                    Serial.updateStatus(res.status);
                } else {
                    // Fallback for early load or error
                    const el = document.getElementById('serial-status-bar');
                    if (el && el.innerText !== res.status) el.innerText = res.status;
                }
            }
        } catch(e) {
            // Polling errors (e.g. backend closed)
        }
    }
    setTimeout(pollSerialLoop, 20); // 50Hz polling
}

function updateMetrics(m) {
    const container = document.getElementById('ssh-metrics-container');
    if(!container) return;
    container.style.display = 'flex';
    lastMetrics = m;

    const cpuEl = document.getElementById('met-cpu');
    cpuEl.innerHTML = `CPU: ${m.cpu_usage.toFixed(0)}% <small style='opacity:0.7'>[${m.highest_cpu_core_usage.toFixed(0)}%]</small>`;
    cpuEl.className = 'metric-item ' + getColorClass(m.cpu_usage);
    
    const ramEl = document.getElementById('met-ram');
    const ramPct = m.ram_total > 0 ? (m.ram_used / m.ram_total) * 100 : 0;
    ramEl.innerText = `RAM: ${ramPct.toFixed(0)}%`;
    ramEl.className = 'metric-item ' + getColorClass(ramPct);

    const diskEl = document.getElementById('met-disk');
    diskEl.innerText = `/: ${m.root_disk_usage.toFixed(0)}%`;
    diskEl.className = 'metric-item ' + getColorClass(m.root_disk_usage);

    const netEl = document.getElementById('met-net');
    netEl.innerText = `⬇${m.net_rx_mbps.toFixed(1)} ⬆${m.net_tx_mbps.toFixed(1)} Mbps`;

    if(!container.hasAttribute('data-events')) {
        container.setAttribute('data-events', 'true');
        cpuEl.onmouseenter = () => showTooltip('cpu');
        ramEl.onmouseenter = () => showTooltip('ram');
        diskEl.onmouseenter = () => showTooltip('disk');
        netEl.onmouseenter = () => showTooltip('net');
        container.onmouseleave = hideTooltip;
    }
}

function getColorClass(val) {
    if(val < 50) return 'metric-safe';
    if(val < 75) return 'metric-warn';
    return 'metric-crit';
}

function showTooltip(type) {
    if(!lastMetrics) return;
    const tt = document.getElementById('metrics-tooltip');
    let html = "";

    if (type === 'ram') {
        const usedGB = (lastMetrics.ram_used / 1024 / 1024 / 1024).toFixed(2);
        const totalGB = (lastMetrics.ram_total / 1024 / 1024 / 1024).toFixed(2);
        const freeGB = ((lastMetrics.ram_total - lastMetrics.ram_used) / 1024 / 1024 / 1024).toFixed(2);
        html = `<strong>RAM Details</strong>
                <div>Used: ${usedGB} GB</div>
                <div>Free: ${freeGB} GB</div>
                <div>Total: ${totalGB} GB</div>`;
    } else if (type === 'disk') {
        html = `<strong>Disk Mounts</strong>`;
        lastMetrics.disk_details.forEach(([mount, usage]) => {
            html += `<div>${mount}: ${usage.toFixed(1)}%</div>`;
        });
    } else if (type === 'cpu') {
        html = `<strong>CPU Details</strong>
                <div>Total Load: ${lastMetrics.cpu_usage.toFixed(1)}%</div>
                <div>Max Core: ${lastMetrics.highest_cpu_core_usage.toFixed(1)}%</div>`;
    } else if (type === 'net') {
        html = `<strong>Network (In/Out)</strong>
                <div>RX: ${lastMetrics.net_rx_mbps.toFixed(2)} Mbps</div>
                <div>TX: ${lastMetrics.net_tx_mbps.toFixed(2)} Mbps</div>`;
    }

    if(html) {
        tt.innerHTML = html;
        tt.style.display = 'block';
    }
}

function hideTooltip() {
    document.getElementById('metrics-tooltip').style.display = 'none';
}

window.listen('debug-log', e => {
    window.AppState.log("[RUST] " + e.payload);
});