// FILE: ui/js/templates/keyboard.js
// ---------------------------------

window.KeyboardRenderer = {
    render: () => {
        const container = document.getElementById('kb-svg-container');
        if (!container) return;

        // 65% Layout Configuration
        const rows = [
            // Row 1 (F-Keys)
            [
                ["Esc", 1], ["F1", 1], ["F2", 1, "key-f2"], ["F3", 1], ["F4", 1], ["F5", 1], ["F6", 1, "key-f6"], ["F7", 1], ["F8", 1], ["F9", 1], ["F10", 1], ["F11", 1], ["F12", 1], ["Del", 1, "key-del"], ["Home", 1]
            ],
            // Row 2 (Numbers)
            [
                ["`", 1], ["1", 1], ["2", 1], ["3", 1], ["4", 1], ["5", 1], ["6", 1], ["7", 1], ["8", 1], ["9", 1], ["0", 1], ["-", 1], ["=", 1], ["Bksp", 2, "key-bksp"]
            ],
            // Row 3 (QWERTY)
            [
                ["Tab", 1.5], ["Q", 1], ["W", 1], ["E", 1], ["R", 1, "key-r"], ["T", 1], ["Y", 1], ["U", 1], ["I", 1], ["O", 1], ["P", 1], ["[", 1], ["]", 1], ["\\", 1.5], ["PgUp", 1]
            ],
            // Row 4 (ASDF)
            [
                ["Caps", 1.75], ["A", 1], ["S", 1, "key-s"], ["D", 1], ["F", 1], ["G", 1], ["H", 1], ["J", 1, "key-j"], ["K", 1], ["L", 1], [";", 1], ["'", 1], ["Enter", 2.25, "key-enter"], ["PgDn", 1]
            ],
            // Row 5 (ZXCV)
            [
                ["Shift", 2.25, "key-shift"], ["Z", 1], ["X", 1], ["C", 1, "key-c"], ["V", 1, "key-v"], ["B", 1], ["N", 1, "key-n"], ["M", 1], [",", 1], [".", 1], ["/", 1], ["Shift", 1.75], ["Up", 1, "key-up"], ["End", 1]
            ],
            // Row 6 (Mods)
            [
                ["Ctrl", 1.25, "key-ctrl"], ["Win", 1.25], ["Alt", 1.25, "key-alt"], ["Space", 6.25], ["Alt", 1], ["Fn", 1], ["Ctrl", 1, "key-ctrl"], ["Left", 1], ["Down", 1, "key-down"], ["Right", 1]
            ]
        ];

        // Detailed Descriptions with Explicit Combos
        const shortcuts = {
            // -- Navigation --
            "key-up": { desc: "(Up Arrow) Move Up\nSelect previous file in list.", mods: [] },
            "key-down": { desc: "(Down Arrow) Move Down\nSelect next file in list.", mods: [] },
            
            "key-f6": { desc: "(F6) Switch Focus\nToggles cursor between the File List and the Editor.", mods: [] },
            "key-enter": { desc: "(Enter) Open\nOpens the selected file or enters the selected folder.", mods: [] },
            "key-bksp": { desc: "(Backspace) Go Up\nNavigates to the parent directory.", mods: [] },

            // -- File Operations --
            "key-s": { desc: "(Ctrl+S) Save File\nUploads changes in the active editor to the remote server.", mods: ["key-ctrl"] },
            "key-r": { desc: "(Ctrl+R) Reload File\nDiscards changes and re-downloads the file from the server.", mods: ["key-ctrl"] },
            "key-c": { desc: "(Ctrl+C) Copy File\nCopies the selected file path to the internal clipboard.", mods: ["key-ctrl"] },
            "key-v": { desc: "(Ctrl+V) Paste File\nPastes the file from internal clipboard to the current folder.", mods: ["key-ctrl"] },
            "key-f2": { desc: "(F2) Rename Item\nStarts an inline rename for the selected file or folder.", mods: [] },
            "key-del": { desc: "(Del) Delete Item\nDeletes the selected file or folder.", mods: [] },
            
            // -- Creation --
            "key-n": { desc: "(Ctrl+Shift+N) New Folder\nCreates a new directory in the current location.", mods: ["key-ctrl", "key-shift"] },
            "key-j": { desc: "(Ctrl+Shift+J) New File\nCreates a new empty file in the current location.", mods: ["key-ctrl", "key-shift"] },
            
            // -- Modifiers --
            "key-ctrl": { 
                desc: "Control Modifier\nUsed for file operations.", 
                highlightAll: true,
                mods: ["key-shift"] 
            },
            "key-alt": { 
                desc: "Alt Modifier\n(Alt + Click) a file to copy to your System Clipboard.", 
                highlightAll: false 
            },
            "key-shift": { 
                desc: "Shift Modifier\nUsed with Ctrl for creating new items.", 
                highlightAll: true, 
                mods: ["key-ctrl"] 
            },
        };

        const keySize = 40;
        const gap = 5;
        let svg = `<svg width="100%" height="280" viewBox="0 0 860 280">`;

        let y = 0;
        rows.forEach((row) => {
            let x = 0;
            row.forEach(([label, widthMult, id]) => {
                const w = (keySize * widthMult) + (gap * (widthMult - 1));
                const h = keySize;
                const keyId = id || `k-${label}`;
                
                const hasInfo = shortcuts[keyId] !== undefined;
                const infoClass = hasInfo ? "has-info" : "";

                // Use data-id on Group for easier selection
                svg += `
                    <g class="kb-group" data-id="${keyId}">
                        <rect x="${x}" y="${y}" width="${w}" height="${h}" rx="4" class="kb-key ${infoClass}" id="rect-${keyId}"></rect>
                        <text x="${x + w/2}" y="${y + h/2}" class="kb-text">${label}</text>
                    </g>
                `;
                x += w + gap;
            });
            y += keySize + gap;
        });
        svg += `</svg>`;
        container.innerHTML = svg;

        const tooltip = document.getElementById('kb-tooltip');
        const groups = container.querySelectorAll('.kb-group');

        groups.forEach(g => {
            g.addEventListener('mouseenter', () => {
                const id = g.dataset.id;
                const info = shortcuts[id];

                document.querySelectorAll('.mod-active, .key-active').forEach(el => {
                    el.classList.remove('mod-active', 'key-active');
                });

                if (info) {
                    tooltip.innerHTML = info.desc.replace(/\n/g, '<br>');
                    tooltip.style.color = "#50fa7b";
                    tooltip.style.fontWeight = "bold";

                    // 1. Highlight Self
                    const allSelfInstances = container.querySelectorAll(`[data-id="${id}"] rect`);
                    allSelfInstances.forEach(rect => {
                        // User Request: Modifiers (Ctrl/Shift/Alt) always Blue
                        if (info.highlightAll) {
                            rect.classList.add('mod-active');
                        } else {
                            rect.classList.add('key-active');
                        }
                    });

                    // 2. Scan for Targets (If we are a modifier)
                    if (info.highlightAll) {
                         Object.entries(shortcuts).forEach(([k, v]) => {
                             if(v.mods && v.mods.includes(id)) {
                                 // Highlight files actions in Green
                                 const targets = container.querySelectorAll(`[data-id="${k}"] rect`);
                                 targets.forEach(t => t.classList.add('key-active'));
                             }
                         });
                    }

                    // 3. Highlight Peers (If we require a modifier)
                    if (info.mods) {
                        info.mods.forEach(modId => {
                            // FIX: querySelectorAll to catch BOTH Ctrl keys
                            const modRects = container.querySelectorAll(`[data-id="${modId}"] rect`);
                            modRects.forEach(r => r.classList.add('mod-active'));
                        });
                    }
                }
            });

            g.addEventListener('mouseleave', () => {
                document.querySelectorAll('.mod-active, .key-active').forEach(el => {
                    el.classList.remove('mod-active', 'key-active');
                });
                tooltip.innerHTML = "Interactive Map<br>Hover over highlighted keys to see available shortcuts.";
                tooltip.style.color = "#aaa";
                tooltip.style.fontWeight = "normal";
            });
        });
    }
};