// FILE: ui/js/tools/sync.js
// --------------------------

const SyncWizard = {
    currentPlan: null, 
    _initialized: false,
    
    ensureInit: () => {
        if (SyncWizard._initialized) return;
        
        const btnAnalyze = document.getElementById('sync-btn-analyze');
        const btnExecute = document.getElementById('sync-btn-execute');
        const btnCancel = document.getElementById('sync-btn-cancel');
        const modal = document.getElementById('sync-wizard-modal');

        if (!modal) return;

        if (btnAnalyze) btnAnalyze.onclick = SyncWizard.analyze;
        if (btnExecute) btnExecute.onclick = SyncWizard.execute;
        if (btnCancel) btnCancel.onclick = SyncWizard.hide;
        
        modal.onclick = (e) => {
            if (e.target === modal) SyncWizard.hide();
        };

        SyncWizard._initialized = true;
    },

    open: async () => {
        SyncWizard.ensureInit(); 

        const modal = document.getElementById('sync-wizard-modal');
        if (!modal) {
            console.error("Sync Modal Template missing.");
            return;
        }

        const selectors = document.querySelectorAll('.sync-host-select');
        let hostOptions = `<option value="" disabled selected>Select Host...</option>`;
        
        try {
            const config = await invoke('get_app_config');
            if (config && config.hosts) {
                const hosts = Object.values(config.hosts).sort((a, b) => 
                    a.name.localeCompare(b.name)
                );
                
                hosts.forEach(h => {
                    hostOptions += `<option value="${h.id}">${h.name} (${h.hostname})</option>`;
                });
            }
        } catch(e) {}

        selectors.forEach(s => s.innerHTML = hostOptions);

        const resultsContainer = document.getElementById('sync-results-container');
        if(resultsContainer) resultsContainer.style.display = 'none';
        
        const list = document.getElementById('sync-list');
        if(list) list.innerHTML = "";
        
        const lp = document.getElementById('sync-local-path');
        if(lp) lp.value = "";
        
        const rp = document.getElementById('sync-remote-path');
        if(rp) rp.value = "";
        
        const ex = document.getElementById('sync-excludes');
        if(ex) ex.value = "node_modules, .git, .DS_Store, target, .vscode"; 
        
        const exeBtn = document.getElementById('sync-btn-execute');
        if(exeBtn) exeBtn.style.display = 'none';
        
        const statusMsg = document.getElementById('sync-status-msg');
        if(statusMsg) statusMsg.innerText = "";
        
        SyncWizard.currentPlan = null;

        modal.style.display = 'flex';
        if(lp) lp.focus();
    },

    hide: () => {
        const modal = document.getElementById('sync-wizard-modal');
        if (modal) modal.style.display = 'none';
    },

    analyze: async () => {
        const localPath = document.getElementById('sync-local-path').value.trim();
        const remoteHost = document.getElementById('sync-remote-host').value;
        const remotePath = document.getElementById('sync-remote-path').value.trim();
        const mode = document.getElementById('sync-mode').value;
        const excludesVal = document.getElementById('sync-excludes').value;
        
        if(!localPath || !remotePath || !remoteHost) {
            alert("Please provide Local Path, Remote Host, and Remote Path.");
            return;
        }

        const excludes = excludesVal.split(/[,\n]/).map(s => s.trim()).filter(s => s.length > 0);
        
        const container = document.getElementById('sync-results-container');
        const list = document.getElementById('sync-list');
        const summary = document.getElementById('sync-summary-text');
        const btnExecute = document.getElementById('sync-btn-execute');
        const statusMsg = document.getElementById('sync-status-msg');
        
        if(statusMsg) statusMsg.innerText = "";
        
        container.style.display = 'flex';
        list.innerHTML = `
            <div style="display:flex; flex-direction:column; align-items:center; justify-content:center; height:100%; color:#888;">
                <div style="font-size:24px; margin-bottom:10px;">🔍</div>
                <div>Scanning file trees...</div>
                <div style="font-size:11px; margin-top:5px; opacity:0.7;">Comparing local vs remote metadata</div>
            </div>
        `;
        summary.innerText = "Analyzing...";
        btnExecute.style.display = 'none';

        try {
            const plan = await invoke('sftp_sync_dry_run', {
                hostId: remoteHost,
                localPath: localPath,
                remotePath: remotePath,
                mode: mode,
                excludes: excludes
            });

            SyncWizard.currentPlan = { 
                hostId: remoteHost,
                localPath: localPath,
                remotePath: remotePath,
                actions: plan.actions 
            };

            SyncWizard.renderResults(plan);
            
        } catch (e) {
            // BATCH 4: Smart Error Display
            let friendlyError = e;
            if (String(e).includes("Local path not found")) friendlyError = "The local directory does not exist.";
            if (String(e).includes("Remote scan failed")) friendlyError = "Could not access remote path. Check permissions.";
            
            list.innerHTML = `<div style="color:#ff5555; padding:20px; text-align:center;">
                <strong>Analysis Failed</strong><br>${friendlyError}
            </div>`;
            summary.innerText = "Error";
        }
    },

    renderResults: (plan) => {
        const list = document.getElementById('sync-list');
        const summary = document.getElementById('sync-summary-text');
        const totalSize = document.getElementById('sync-total-size');
        const btnExecute = document.getElementById('sync-btn-execute');

        list.innerHTML = "";

        const actions = plan.actions;
        if (actions.length === 0) {
            list.innerHTML = `
                <div style="display:flex; flex-direction:column; align-items:center; justify-content:center; height:100%; color:#50fa7b;">
                    <div style="font-size:24px; margin-bottom:10px;">✨</div>
                    <div>Everything is in sync.</div>
                </div>`;
            summary.innerText = "Synced";
            btnExecute.style.display = 'none';
            return;
        }

        const upCount = actions.filter(a => a.direction === 'Upload').length;
        const downCount = actions.filter(a => a.direction === 'Download').length;
        const delCount = actions.filter(a => a.direction === 'DeleteRemote').length;

        summary.innerText = `Pending: ${upCount} Uploads, ${downCount} Downloads, ${delCount} Deletions`;
        
        const fmt = (typeof SftpUtils !== 'undefined') ? SftpUtils.formatShortSize : (b) => b + "B";
        totalSize.innerText = `Traffic: ↑${fmt(plan.total_up_bytes)} ↓${fmt(plan.total_down_bytes)}`;
        
        let html = '<table style="width:100%; border-collapse:collapse; font-size:12px; font-family:\'Segoe UI\';">';
        
        actions.forEach((f, idx) => {
            let color = "#fff";
            let icon = "❓";
            let actionText = f.direction;
            
            if (f.direction === "Upload") {
                color = f.reason === "New" ? "#50fa7b" : "#f1fa8c"; 
                icon = "⬆️";
            } else if (f.direction === "Download") {
                color = "#8be9fd";
                icon = "⬇️";
            } else if (f.direction === "DeleteRemote") {
                color = "#ff5555";
                icon = "🗑️";
                actionText = "Delete";
            } else if (f.direction === "CreateDir") {
                color = "#bd93f9";
                icon = "📁";
                actionText = "New Dir";
            }

            const bg = idx % 2 === 0 ? 'transparent' : '#1e1e1e';

            html += `
                <tr style="background-color:${bg}; border-bottom:1px solid #2a2a2a;">
                    <td style="padding:6px 10px; width:30px; text-align:center;">${icon}</td>
                    <td style="padding:6px 10px; color:#ddd;">
                        <div style="font-weight:bold; color:${color};">${actionText}</div>
                    </td>
                    <td style="padding:6px 10px; width:100%;">
                        <div style="color:#eee; word-break:break-all;">${f.rel_path}</div>
                    </td>
                    <td style="padding:6px 10px; white-space:nowrap; color:#888; text-align:right;">
                        ${f.reason}
                    </td>
                </tr>
            `;
        });
        html += '</table>';
        list.innerHTML = html;

        btnExecute.style.display = 'inline-block';
        btnExecute.disabled = false;
        btnExecute.innerText = `Execute ${actions.length} Changes`;
    },

    execute: async () => {
        if (!SyncWizard.currentPlan || SyncWizard.currentPlan.actions.length === 0) return;

        const count = SyncWizard.currentPlan.actions.length;
        if (!confirm(`Are you sure you want to apply ${count} changes?`)) return;

        const btn = document.getElementById('sync-btn-execute');
        const list = document.getElementById('sync-list');
        const statusMsg = document.getElementById('sync-status-msg');
        
        btn.disabled = true;
        btn.innerText = "Syncing...";
        statusMsg.innerText = "Executing... Do not close this window.";
        
        list.innerHTML = `
            <div style="display:flex; flex-direction:column; align-items:center; justify-content:center; height:100%; color:#007acc;">
                <div style="font-size:24px; margin-bottom:10px;">⏳</div>
                <div>Applying ${count} changes...</div>
                <div style="font-size:11px; margin-top:5px; color:#888;">This may take a while for large files.</div>
            </div>
        `;

        try {
            await invoke('sftp_sync_execute', {
                hostId: SyncWizard.currentPlan.hostId,
                localBase: SyncWizard.currentPlan.localPath,
                remoteBase: SyncWizard.currentPlan.remotePath,
                actions: SyncWizard.currentPlan.actions
            });

            list.innerHTML = `
                <div style="display:flex; flex-direction:column; align-items:center; justify-content:center; height:100%; color:#50fa7b;">
                    <div style="font-size:32px; margin-bottom:10px;">✅</div>
                    <div style="font-size:14px; font-weight:bold;">Sync Complete!</div>
                    <div style="font-size:12px; margin-top:5px; opacity:0.8;">${count} actions performed successfully.</div>
                </div>`;
            
            btn.style.display = 'none';
            document.getElementById('sync-summary-text').innerText = "Finished";
            statusMsg.innerText = "";

        } catch (e) {
            // BATCH 4: Smart Execution Diagnostics
            let msg = String(e);
            let tip = "";
            
            if (msg.includes("Permission denied")) {
                tip = "<br>Tip: Check ownership of the remote folder.";
            } else if (msg.includes("Connection lost") || msg.includes("Channel closed")) {
                tip = "<br>Tip: The SSH connection may have timed out.";
            }

            list.innerHTML = `<div style="color:#ff5555; padding:20px; text-align:center;">
                <strong>Execution Error</strong><br>${msg}${tip}
            </div>`;
            btn.innerText = "Retry";
            btn.disabled = false;
            statusMsg.innerText = "Error Occurred";
        }
    }
};